const express = require('express');
const cors = require('cors');
const axios = require('axios');
const TokenManager = require('./token-manager');

const app = express();
const PORT = process.env.PORT || 3000;
const API_BASE_URL = 'https://api.digitalsevaonline.com/api/admin/state-urls';

app.use(cors());
app.use(express.json());

// Helper: Fetch Config from Backend
async function getStateConfig(stateName) {
    try {
        const response = await axios.get(`${API_BASE_URL}/fetch-config`, {
            params: { state_name: stateName }
        });
        if (response.data.success) {
            return response.data.data;
        }
    } catch (error) {
        if (error.response) {
            console.error(`Error fetching config for ${stateName}:`, error.response.status, error.response.data);
        } else {
            console.error(`Error fetching config for ${stateName}:`, error.message);
        }
    }
    return null;
}

// Helper: Update Token in Backend
async function updateBackendToken(stateName, token, credentials = {}) {
    try {
        await axios.post(`${API_BASE_URL}/update-token`, {
            state_name: stateName,
            token: token,
            ...credentials
        });
        console.log(`Token updated in backend for ${stateName}`);
    } catch (error) {
        console.error(`Error updating backend token for ${stateName}:`, error.message);
    }
}

// Verification Endpoint
app.get('/verify_card_details', async (req, res) => {
    const { state_name, aadhar } = req.query;

    if (!state_name || !aadhar) {
        return res.status(400).json({ error: 'Missing state_name or aadhar parameter' });
    }

    console.log(`[API] Processing request for State: ${state_name}, Aadhaar: ${aadhar}`);

    // 1. Get Config for State
    const config = await getStateConfig(state_name);
    if (!config) {
        return res.status(404).json({ error: 'State configuration not found' });
    }

    // 2. Instantiate TokenManager for this State
    const tokenManager = new TokenManager(state_name);

    // Pre-load token from config (if available and not expired in DB)
    if (config.token && !tokenManager.token) {
        tokenManager.setToken(config.token);
    }

    if (!config.api_url) {
        return res.status(500).json({ error: 'State API URL not configured' });
    }

    const url = `${config.api_url}?aadhaarNumber=${encodeURIComponent(aadhar)}`;

    const makeRequest = async (token, isRetry = false) => {
        try {
            const response = await axios.get(url, {
                headers: {
                    'accept': 'application/json, text/plain, */*',
                    'accept-encoding': 'gzip, deflate, br, zstd',
                    'accept-language': 'en-GB,en-US;q=0.9,en;q=0.8',
                    'authorization': `Bearer ${token}`,
                    'dnt': '1',
                    'referer': config.url,
                    'sec-fetch-dest': 'empty',
                    'sec-fetch-mode': 'cors',
                    'sec-fetch-site': 'same-origin',
                    'user-agent': 'Mozilla/5.0 (iPhone; CPU iPhone OS 16_6 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/16.6 Mobile/15E148 Safari/604.1'
                }
            });
            return { success: true, response };
        } catch (error) {
            if (error.response && (error.response.status === 401 || error.response.status === 403) && !isRetry) {
                console.log(`[API] Token expired for ${state_name}, refreshing...`);
                return { success: false, retry: true, error };
            }
            throw error;
        }
    };

    try {
        // 3. Get Valid Token (Initial Attempt)
        let token = await tokenManager.getValidToken({
            url: config.url,
            includes_pattern: config.includes_pattern,
            username: config.username,
            password: config.password
        });

        // If newly refreshed, update backend
        if (token && token !== config.token) {
            await updateBackendToken(state_name, token);
        }

        // 4. Call AgriStack API
        let result = await makeRequest(token);

        // Handle Retry Logic
        if (!result.success && result.retry) {
            console.log(`[API] Retrying request for ${state_name} with forced refresh...`);

            // Force Refresh Token
            await tokenManager.refreshToken({
                url: config.url,
                includes_pattern: config.includes_pattern,
                username: config.username,
                password: config.password
            });

            const newToken = tokenManager.token;
            if (newToken) {
                await updateBackendToken(state_name, newToken);
                result = await makeRequest(newToken, true);
            } else {
                throw new Error("Failed to obtain a new token during retry");
            }
        }

        const response = result.response;

        if (!response.data || response.data.status === false || response.data.data?.status === "failed") {
            return res.status(404).json({
                error: 'Farmer not found',
                details: response.data?.data?.message || 'No farmer found',
                apiResponse: response.data
            });
        }

        const farmerDetails = response.data.data.farmerDetails;
        const landOwnerShips = response.data.data.farmerLandOwnerShips || [];

        const processedData = {
            status: response.status,
            http_code: response.status,
            data: {
                farmerID: farmerDetails.centralId || null,
                farmerAadharNumber: encodeURIComponent(aadhar) || null,
                farmerDob: farmerDetails.farmerDob || null,
                farmerNameEn: farmerDetails.farmerNameEn || null,
                farmerNameLocal: farmerDetails.farmerNameLocal || null,
                farmerGenederType: farmerDetails.farmerGenederType?.genderDescEng || null,
                farmerPinCode: farmerDetails.farmerPinCode || null,
                farmerAddressEn: farmerDetails.farmerAddressEn || null,
                farmerMobileNumber: farmerDetails.farmerMobileNumber || null,
                ferFarmerPhotograph: farmerDetails.farmerExtendedRegistry?.ferFarmerPhotograph || null,
                farmerLandOwnerShips: landOwnerShips.map(land => ({
                    extentAssignedAreaInHectare: land.extentAssignedAreaInHectare || null,
                    villageName: land.farmlandPlotRegistryId?.villageLgdMaster?.villageName || null,
                    districtName: land.farmlandPlotRegistryId?.villageLgdMaster?.districtLgdCode?.districtName || null,
                    subDistrictName: land.farmlandPlotRegistryId?.villageLgdMaster?.subDistrictLgdCode?.subDistrictName || null,
                    surveyNumber: land.farmlandPlotRegistryId?.surveyNumber || null
                }))
            }
        };

        res.status(200).json(processedData);

    } catch (error) {
        // Handle 401/403 - Token Expired
        if (error.response && (error.response.status === 401 || error.response.status === 403)) {
            console.log(`[API] Token expired for ${state_name}, refreshing...`);
            try {
                // Force Refresh
                await tokenManager.refreshToken({
                    url: config.url,
                    includes_pattern: config.includes_pattern,
                    username: config.username,
                    password: config.password
                });

                const newToken = tokenManager.token;
                await updateBackendToken(state_name, newToken);

                // Retry Request
                const retryResponse = await axios.get(url, {
                    headers: {
                        'authorization': `Bearer ${newToken}`,
                        'referer': config.url,
                        'user-agent': 'Mozilla/5.0 (iPhone; CPU iPhone OS 16_6 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/16.6 Mobile/15E148 Safari/604.1'
                    }
                });

                // Logic repeated for brevity - ideally separate function
                const farmerDetails = retryResponse.data.data.farmerDetails;
                const landOwnerShips = retryResponse.data.data.farmerLandOwnerShips || [];

                return res.status(200).json({
                    status: retryResponse.status,
                    data: {
                        farmerID: farmerDetails.centralId || null,
                        farmerNameEn: farmerDetails.farmerNameEn || null,
                        // ... (rest of fields)
                    }
                });

            } catch (refreshError) {
                console.error(`[API] Refresh failed: ${refreshError.message}`);
                return res.status(500).json({ error: 'Authentication failed after retry' });
            }
        }

        console.error(`[API] Error: ${error.message}`);
        res.status(500).json({ error: error.message });
    }
});

app.listen(PORT, () => {
    console.log(`CSC Automation Server running on port ${PORT}`);
});
