<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class RechargeOffer extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'description',
        'minimum_recharge_amount',
        'discount_type',
        'discount_value',
        'cashback_amount',
        'validity_start_date',
        'validity_end_date',
        'is_active',
        'created_by',
    ];

    protected $casts = [
        'minimum_recharge_amount' => 'decimal:2',
        'discount_value' => 'decimal:2',
        'cashback_amount' => 'decimal:2',
        'validity_start_date' => 'date',
        'validity_end_date' => 'date',
        'is_active' => 'boolean',
    ];

    /**
     * Get the admin who created this offer
     */
    public function admin()
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    /**
     * Check if offer is currently valid
     */
    public function isValid()
    {
        $today = now()->toDateString();
        return $this->is_active 
            && $this->validity_start_date <= $today 
            && $this->validity_end_date >= $today;
    }

    /**
     * Calculate discount for a given recharge amount
     */
    public function calculateDiscount($rechargeAmount)
    {
        if (!$this->discount_type || !$this->discount_value) {
            return 0;
        }

        if ($this->discount_type === 'percentage') {
            return ($rechargeAmount * $this->discount_value) / 100;
        } else {
            return $this->discount_value;
        }
    }

    /**
     * Calculate total benefit for a given recharge amount
     */
    public function calculateTotalBenefit($rechargeAmount)
    {
        $discount = $this->calculateDiscount($rechargeAmount);
        $cashback = $this->cashback_amount ?? 0;
        return $discount + $cashback;
    }
}
