<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Services\ReferralService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class UserAuthController extends Controller
{
    /**
     * User Signup - Sends OTP
     */
    public function signup(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|unique:users,email',
            'phone_number' => 'required|string|unique:users,phone_number',
            'user_type' => 'nullable|in:user,distributor',
            'referral_code' => 'nullable|string|exists:users,referral_code',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Generate 6-digit OTP
        $otp = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
        $otpExpiresAt = now()->addMinutes(10); // OTP valid for 10 minutes

        // Check if user already exists (unverified)
        $user = User::where('email', $request->email)
            ->orWhere('phone_number', $request->phone_number)
            ->first();

        if ($user && $user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User already exists and is verified'
            ], 409);
        }

        if ($user) {
            // Update existing unverified user
            $user->update([
                'email' => $request->email,
                'phone_number' => $request->phone_number,
                'user_type' => $request->user_type ?? 'user',
                'otp' => $otp,
                'otp_expires_at' => $otpExpiresAt,
                'is_verified' => false,
            ]);
        } else {
            // Create new user
            $user = User::create([
                'email' => $request->email,
                'phone_number' => $request->phone_number,
                'user_type' => $request->user_type ?? 'user',
                'wallet_balance' => 0.00,
                'otp' => $otp,
                'otp_expires_at' => $otpExpiresAt,
                'is_verified' => false,
            ]);
        }

        // Send OTP via SMS
        $smsSent = $this->sendSmsOtp($user->phone_number, $otp);

        return response()->json([
            'success' => true,
            'message' => $smsSent ? 'OTP sent successfully to your phone number' : 'OTP generated but SMS sending failed. Please try resending OTP.',
            'data' => [
                'user_id' => $user->id,
                'email' => $user->email,
                'phone_number' => $user->phone_number,
                // OTP is not returned for security
            ]
        ], 200);
    }

    /**
     * Verify OTP and Complete Signup
     * Password is created during OTP verification (stored in plain text)
     */
    public function verifyOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'otp' => 'required|string|size:6',
            'password' => 'required|string|min:6',
            'referral_code' => 'nullable|string|exists:users,referral_code',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        if ($user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User already verified'
            ], 400);
        }

        // Check if OTP matches
        if ($user->otp !== $request->otp) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid OTP'
            ], 401);
        }

        // Check if OTP has expired
        if ($user->otp_expires_at && $user->otp_expires_at->isPast()) {
            return response()->json([
                'success' => false,
                'message' => 'OTP has expired. Please request a new one.'
            ], 401);
        }

        // Verify user and create password (plain text, not encrypted)
        $user->update([
            'is_verified' => true,
            'password' => $request->password, // Plain text password
            'otp' => null,
            'otp_expires_at' => null,
            'email_verified_at' => now(),
        ]);

        // Process referral if referral code provided
        if ($request->has('referral_code') && $request->referral_code) {
            ReferralService::processSignupReferral($user, $request->referral_code);
        } else {
            // Generate referral code for user if not exists
            if (!$user->referral_code) {
                $user->referral_code = User::generateReferralCode();
                $user->save();
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Signup completed successfully',
            'data' => [
                'id' => $user->id,
                'email' => $user->email,
                'phone_number' => $user->phone_number,
                'user_type' => $user->user_type,
                'wallet_balance' => $user->fresh()->wallet_balance,
                'is_verified' => $user->is_verified,
                'referral_code' => $user->fresh()->referral_code,
            ]
        ], 200);
    }

    /**
     * User Login
     * Login with email and password (password stored in plain text)
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        if (!$user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User account is not verified. Please complete signup first.'
            ], 403);
        }

        // Check password in plain text (not encrypted)
        if ($user->password !== $request->password) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        // Create token for user
        $token = $user->createToken('user-token')->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'data' => [
                'id' => $user->id,
                'email' => $user->email,
                'phone_number' => $user->phone_number,
                'user_type' => $user->user_type,
                'wallet_balance' => $user->wallet_balance,
                'token' => $token,
            ]
        ], 200);
    }

    /**
     * Forget Password - Request OTP for password reset
     */
    public function forgetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            // For security, don't reveal if user exists or not
            return response()->json([
                'success' => true,
                'message' => 'If the email exists, an OTP has been sent to your registered phone number.'
            ], 200);
        }

        if (!$user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User account is not verified. Please complete signup first.'
            ], 403);
        }

        // Generate 6-digit OTP for password reset
        $otp = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
        $otpExpiresAt = now()->addMinutes(10); // OTP valid for 10 minutes

        // Update user with password reset OTP
        $user->update([
            'otp' => $otp,
            'otp_expires_at' => $otpExpiresAt,
        ]);

        // Send OTP via SMS
        $smsSent = $this->sendPasswordResetSmsOtp($user->phone_number, $otp);

        return response()->json([
            'success' => true,
            'message' => $smsSent ? 'OTP sent successfully to your registered phone number for password reset' : 'OTP generated but SMS sending failed. Please try again.',
            'data' => [
                'email' => $user->email,
                // OTP is not returned for security
            ]
        ], 200);
    }

    /**
     * Reset Password - Verify OTP and set new password
     */
    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'otp' => 'required|string|size:6',
            'password' => 'required|string|min:6',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        if (!$user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User account is not verified. Please complete signup first.'
            ], 403);
        }

        // Check if OTP matches
        if ($user->otp !== $request->otp) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid OTP'
            ], 401);
        }

        // Check if OTP has expired
        if ($user->otp_expires_at && $user->otp_expires_at->isPast()) {
            return response()->json([
                'success' => false,
                'message' => 'OTP has expired. Please request a new one.'
            ], 401);
        }

        // Reset password (plain text, not encrypted - consistent with existing system)
        $user->update([
            'password' => $request->password, // Plain text password
            'otp' => null,
            'otp_expires_at' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password reset successfully. You can now login with your new password.',
            'data' => [
                'id' => $user->id,
                'email' => $user->email,
            ]
        ], 200);
    }

    /**
     * Resend OTP
     */
    public function resendOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        if ($user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User already verified'
            ], 400);
        }

        // Generate new OTP
        $otp = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
        $otpExpiresAt = now()->addMinutes(10);

        $user->update([
            'otp' => $otp,
            'otp_expires_at' => $otpExpiresAt,
        ]);

        // Send OTP via SMS
        $smsSent = $this->sendSmsOtp($user->phone_number, $otp);

        return response()->json([
            'success' => true,
            'message' => $smsSent ? 'OTP resent successfully to your phone number' : 'OTP generated but SMS sending failed. Please try again.'
        ], 200);
    }

    /**
     * Send OTP via SMS
     */
    private function sendSmsOtp($phoneNumber, $otp)
    {
        $originalPhone = $phoneNumber;
        $phone = null;
        
        try {
            // Remove any non-numeric characters from phone number
            $phone = preg_replace('/[^0-9]/', '', $phoneNumber);
            
            // Ensure phone number starts with country code 91
            if (substr($phone, 0, 2) !== '91') {
                $phone = '91' . $phone;
            }

            $smsMessage = "To verify your Farmer ID account, use OTP {$otp} This code will expire in 10 minutes.";
            
            $smsUrl = "https://sms.webtextsolution.com/sms-panel/api/http/index.php";
            
            $params = [
                'username' => 'INFICOM',
                'apikey' => 'D2180-0FD53',
                'apirequest' => 'Text',
                'sender' => 'FARMID',
                'mobile' => $phone,
                'message' => $smsMessage,
                'route' => 'DND',
                'TemplateID' => '1707176425030137312',
                'format' => 'JSON'
            ];

            Log::info('Attempting to send SMS OTP', [
                'original_phone' => $originalPhone,
                'formatted_phone' => $phone,
                'otp_length' => strlen($otp),
                'sms_provider' => 'WebTextSolution',
                'timestamp' => now()->toDateTimeString()
            ]);
            
            $response = Http::timeout(30)->get($smsUrl, $params);

            if ($response->successful()) {
                $result = $response->json();
                
                // Check if SMS was actually sent successfully
                $isSuccess = isset($result['status']) && $result['status'] === 'success';
                
                if ($isSuccess) {
                    Log::info('SMS OTP sent successfully', [
                        'phone' => $phone,
                        'original_phone' => $originalPhone,
                        'message_id' => $result['message-id'][0] ?? 'N/A',
                        'provider_response' => $result,
                        'sms_provider' => 'WebTextSolution',
                        'timestamp' => now()->toDateTimeString()
                    ]);
                    return true;
                } else {
                    Log::warning('SMS API returned non-success status', [
                        'phone' => $phone,
                        'original_phone' => $originalPhone,
                        'provider_response' => $result,
                        'sms_provider' => 'WebTextSolution',
                        'timestamp' => now()->toDateTimeString()
                    ]);
                    return false;
                }
            } else {
                Log::error('Failed to send SMS OTP - HTTP error', [
                    'phone' => $phone,
                    'original_phone' => $originalPhone,
                    'http_status' => $response->status(),
                    'response_body' => $response->body(),
                    'sms_provider' => 'WebTextSolution',
                    'timestamp' => now()->toDateTimeString()
                ]);
                return false;
            }
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('SMS OTP - Connection exception', [
                'phone' => $phone ?? $originalPhone,
                'original_phone' => $originalPhone,
                'error' => $e->getMessage(),
                'sms_provider' => 'WebTextSolution',
                'timestamp' => now()->toDateTimeString()
            ]);
            return false;
        } catch (\Exception $e) {
            Log::error('SMS OTP - Unexpected exception', [
                'phone' => $phone ?? $originalPhone,
                'original_phone' => $originalPhone,
                'error' => $e->getMessage(),
                'exception_type' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'sms_provider' => 'WebTextSolution',
                'timestamp' => now()->toDateTimeString()
            ]);
            return false;
        }
    }

    /**
     * Send Password Reset OTP via SMS
     */
    private function sendPasswordResetSmsOtp($phoneNumber, $otp)
    {
        $originalPhone = $phoneNumber;
        $phone = null;
        
        try {
            // Remove any non-numeric characters from phone number
            $phone = preg_replace('/[^0-9]/', '', $phoneNumber);
            
            // Ensure phone number starts with country code 91
            if (substr($phone, 0, 2) !== '91') {
                $phone = '91' . $phone;
            }

            $smsMessage = "To verify your Farmer ID account, use OTP {$otp} This code will expire in 10 minutes.";
            
            $smsUrl = "https://sms.webtextsolution.com/sms-panel/api/http/index.php";
            
            $params = [
                'username' => 'INFICOM',
                'apikey' => 'D2180-0FD53',
                'apirequest' => 'Text',
                'sender' => 'FARMID',
                'mobile' => $phone,
                'message' => $smsMessage,
                'route' => 'DND',
                'TemplateID' => '1707176425030137312',
                'format' => 'JSON'
            ];

            Log::info('Attempting to send Password Reset SMS OTP', [
                'original_phone' => $originalPhone,
                'formatted_phone' => $phone,
                'otp_length' => strlen($otp),
                'sms_provider' => 'WebTextSolution',
                'timestamp' => now()->toDateTimeString()
            ]);
            
            $response = Http::timeout(30)->get($smsUrl, $params);

            if ($response->successful()) {
                $result = $response->json();
                
                // Check if SMS was actually sent successfully
                $isSuccess = isset($result['status']) && $result['status'] === 'success';
                
                if ($isSuccess) {
                    Log::info('Password Reset SMS OTP sent successfully', [
                        'phone' => $phone,
                        'original_phone' => $originalPhone,
                        'message_id' => $result['message-id'][0] ?? 'N/A',
                        'provider_response' => $result,
                        'sms_provider' => 'WebTextSolution',
                        'timestamp' => now()->toDateTimeString()
                    ]);
                    return true;
                } else {
                    Log::warning('Password Reset SMS API returned non-success status', [
                        'phone' => $phone,
                        'original_phone' => $originalPhone,
                        'provider_response' => $result,
                        'sms_provider' => 'WebTextSolution',
                        'timestamp' => now()->toDateTimeString()
                    ]);
                    return false;
                }
            } else {
                Log::error('Failed to send Password Reset SMS OTP - HTTP error', [
                    'phone' => $phone,
                    'original_phone' => $originalPhone,
                    'http_status' => $response->status(),
                    'response_body' => $response->body(),
                    'sms_provider' => 'WebTextSolution',
                    'timestamp' => now()->toDateTimeString()
                ]);
                return false;
            }
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('Password Reset SMS OTP - Connection exception', [
                'phone' => $phone ?? $originalPhone,
                'original_phone' => $originalPhone,
                'error' => $e->getMessage(),
                'sms_provider' => 'WebTextSolution',
                'timestamp' => now()->toDateTimeString()
            ]);
            return false;
        } catch (\Exception $e) {
            Log::error('Password Reset SMS OTP - Unexpected exception', [
                'phone' => $phone ?? $originalPhone,
                'original_phone' => $originalPhone,
                'error' => $e->getMessage(),
                'exception_type' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'sms_provider' => 'WebTextSolution',
                'timestamp' => now()->toDateTimeString()
            ]);
            return false;
        }
    }

    /**
     * CSC User Signup
     * Signup with CSC ID, Password, Phone, State (No OTP)
     */
    public function signupWithCsc(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'csc_id' => 'required|string|unique:users,csc_id',
            'password' => 'required|string|min:6',
            'phone_number' => 'required|string|unique:users,phone_number',
            'state' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Create new user with verified status (No OTP for CSC flow)
        $user = User::create([
            'csc_id' => $request->csc_id,
            'password' => $request->password, // Store as plain text as requested
            'phone_number' => $request->phone_number,
            'state' => $request->state,
            'email' => null, // Email is nullable for CSC flow
            'user_type' => 'user',
            'wallet_balance' => 0.00,
            'is_verified' => true,
            'email_verified_at' => now(),
        ]);

        // Generate referral code
        $user->referral_code = User::generateReferralCode();
        $user->save();

        // Create token
        $token = $user->createToken('authToken')->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Signup completed successfully',
            'data' => [
                'id' => $user->id,
                'csc_id' => $user->csc_id,
                'phone_number' => $user->phone_number,
                'wallet_balance' => $user->wallet_balance,
                'token' => $token,
            ]
        ], 200);
    }

    /**
     * CSC User Login
     * Login with CSC ID and Password
     */
    public function loginWithCsc(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'csc_id' => 'required|string',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('csc_id', $request->csc_id)->first();

        if (!$user) {
             return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        // Check if user is verified
        if (!$user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User account is not verified. Please contact admin.'
            ], 403);
        }

        // Check plain text password comparison
        if ($user->password !== $request->password) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        // Create token
        $token = $user->createToken('authToken')->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'data' => [
                'id' => $user->id,
                'csc_id' => $user->csc_id,
                'phone_number' => $user->phone_number,
                'wallet_balance' => $user->wallet_balance,
                'token' => $token,
            ]
        ], 200);
    }
}
