<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\MoneyTransfer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\PersonalAccessToken;

class TransferController extends Controller
{
    /**
     * Get authenticated user from token
     */
    private function getUser(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $user = $accessToken->tokenable;
        if (!$user || !($user instanceof User)) {
            return null;
        }

        return $user;
    }

    /**
     * Search Users for Transfer
     * Distributor can search users by name, email, or phone number
     */
    public function searchUsers(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        // Only distributors can transfer money
        if ($user->user_type !== 'distributor') {
            return response()->json([
                'success' => false,
                'message' => 'Only distributors can transfer money.'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'search' => 'required|string|min:2',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $search = $request->search;

        // Search users by name, email, or phone number
        $users = User::where('id', '!=', $user->id) // Exclude current user
            ->where('is_verified', true) // Only verified users
            ->where(function($query) use ($search) {
                $query->where('name', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%")
                      ->orWhere('phone_number', 'like', "%{$search}%");
            })
            ->select('id', 'name', 'email', 'phone_number', 'user_type')
            ->limit(20)
            ->get();

        return response()->json([
            'success' => true,
            'count' => $users->count(),
            'data' => $users->map(function ($user) {
                return [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone_number' => $user->phone_number,
                    'user_type' => $user->user_type,
                ];
            })
        ], 200);
    }

    /**
     * Transfer Money
     * Distributor transfers money to another user
     */
    public function transfer(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        // Only distributors can transfer money
        if ($user->user_type !== 'distributor') {
            return response()->json([
                'success' => false,
                'message' => 'Only distributors can transfer money.'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'to_user_id' => 'required|exists:users,id',
            'amount' => 'required|numeric|min:0.01',
            'note' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $toUser = User::find($request->to_user_id);

        if (!$toUser) {
            return response()->json([
                'success' => false,
                'message' => 'Recipient user not found'
            ], 404);
        }

        if ($toUser->id === $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'You cannot transfer money to yourself'
            ], 400);
        }

        if (!$toUser->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'Recipient user is not verified'
            ], 400);
        }

        // Check if distributor has sufficient balance
        if ($user->wallet_balance < $request->amount) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient wallet balance',
                'data' => [
                    'required_amount' => (float) $request->amount,
                    'current_balance' => (float) $user->wallet_balance,
                    'shortage' => (float) ($request->amount - $user->wallet_balance)
                ]
            ], 400);
        }

        // Perform transfer using database transaction
        DB::transaction(function () use ($user, $toUser, $request) {
            // Deduct from distributor's wallet
            $user->wallet_balance -= $request->amount;
            $user->save();

            // Add to recipient's wallet
            $toUser->wallet_balance += $request->amount;
            $toUser->save();

            // Create transfer record
            MoneyTransfer::create([
                'from_user_id' => $user->id,
                'to_user_id' => $toUser->id,
                'amount' => $request->amount,
                'note' => $request->note,
            ]);
        });

        return response()->json([
            'success' => true,
            'message' => 'Money transferred successfully',
            'data' => [
                'transfer_id' => MoneyTransfer::latest()->first()->id,
                'from_user' => [
                    'id' => $user->id,
                    'email' => $user->email,
                    'name' => $user->name,
                ],
                'to_user' => [
                    'id' => $toUser->id,
                    'email' => $toUser->email,
                    'name' => $toUser->name,
                ],
                'amount' => (float) $request->amount,
                'note' => $request->note,
                'remaining_balance' => (float) $user->fresh()->wallet_balance,
                'transferred_at' => now(),
            ]
        ], 200);
    }

    /**
     * Get Transfer History
     * Get all transfers made by the distributor
     */
    public function history(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        // Only distributors can view transfer history
        if ($user->user_type !== 'distributor') {
            return response()->json([
                'success' => false,
                'message' => 'Only distributors can view transfer history.'
            ], 403);
        }

        $query = MoneyTransfer::where('from_user_id', $user->id)
            ->with(['toUser:id,name,email,phone_number']);

        // Filter by date range
        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        // Sort options
        $sortBy = $request->input('sort_by', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->input('per_page', 15);
        $perPage = min(max($perPage, 1), 100);

        $transfers = $query->paginate($perPage);

        // Calculate summary
        $summaryQuery = MoneyTransfer::where('from_user_id', $user->id);
        if ($request->has('start_date')) {
            $summaryQuery->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $summaryQuery->whereDate('created_at', '<=', $request->end_date);
        }

        $summary = [
            'total_transfers' => $summaryQuery->count(),
            'total_amount_transferred' => (float) $summaryQuery->sum('amount'),
        ];

        return response()->json([
            'success' => true,
            'summary' => $summary,
            'data' => $transfers->items(),
            'pagination' => [
                'current_page' => $transfers->currentPage(),
                'per_page' => $transfers->perPage(),
                'total' => $transfers->total(),
                'last_page' => $transfers->lastPage(),
            ]
        ], 200);
    }
}

