<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\TopUpRequest;
use App\Models\Upi;
use App\Models\User;
use App\Models\RechargeOffer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TopUpRequestController extends Controller
{
    /**
     * Request Top-Up
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'upi_id' => 'required|exists:upis,id',
            'amount' => 'required|numeric|min:0.01',
            'recharge_offer_id' => 'nullable|exists:recharge_offers,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::findOrFail($request->user_id);

        if (!$user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User is not verified. Please complete signup first.'
            ], 403);
        }

        $upi = Upi::findOrFail($request->upi_id);

        if ($upi->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Selected UPI is not active'
            ], 400);
        }

        // Check if user has a pending request
        $pendingRequest = TopUpRequest::where('user_id', $user->id)
            ->where('status', 'pending')
            ->first();

        if ($pendingRequest) {
            return response()->json([
                'success' => false,
                'message' => 'You already have a pending top-up request. Please wait for admin approval.'
            ], 400);
        }

        $rechargeAmount = $request->amount;
        $discountAmount = 0;
        $cashbackAmount = 0;
        $finalCreditedAmount = $rechargeAmount;
        $offer = null;

        // Process offer if provided
        if ($request->has('recharge_offer_id') && $request->recharge_offer_id) {
            $offer = RechargeOffer::find($request->recharge_offer_id);

            if (!$offer) {
                return response()->json([
                    'success' => false,
                    'message' => 'Recharge offer not found'
                ], 404);
            }

            // Validate offer is active and valid
            if (!$offer->isValid()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Selected offer is not currently valid or active'
                ], 400);
            }

            // Validate minimum recharge amount
            if ($rechargeAmount < $offer->minimum_recharge_amount) {
                return response()->json([
                    'success' => false,
                    'message' => "Minimum recharge amount for this offer is ₹{$offer->minimum_recharge_amount}"
                ], 400);
            }

            // Calculate discount
            $discountAmount = $offer->calculateDiscount($rechargeAmount);
            
            // Get cashback
            $cashbackAmount = $offer->cashback_amount ?? 0;

            // Final credited amount = recharge amount + discount + cashback
            // (Discount is treated as bonus credit, cashback is extra)
            $finalCreditedAmount = $rechargeAmount + $discountAmount + $cashbackAmount;
        }

        $topUpRequest = TopUpRequest::create([
            'user_id' => $user->id,
            'upi_id' => $upi->id,
            'recharge_offer_id' => $offer ? $offer->id : null,
            'amount' => $rechargeAmount,
            'discount_amount' => $discountAmount,
            'cashback_amount' => $cashbackAmount,
            'final_credited_amount' => $finalCreditedAmount,
            'status' => 'verifying_payment', // Set to verifying_payment for SMS-based auto-verification
        ]);

        $responseData = [
            'request_id' => $topUpRequest->id,
            'user_id' => $topUpRequest->user_id,
            'upi_id' => $topUpRequest->upi_id,
            'upi_details' => [
                'upi_id' => $upi->upi_id,
                'name' => $upi->name,
            ],
            'amount' => $topUpRequest->amount,
            'status' => $topUpRequest->status,
            'created_at' => $topUpRequest->created_at,
        ];

        // Add offer details if offer was applied
        if ($offer) {
            $responseData['offer_applied'] = [
                'offer_id' => $offer->id,
                'offer_title' => $offer->title,
                'discount_amount' => $discountAmount,
                'cashback_amount' => $cashbackAmount,
                'final_credited_amount' => $finalCreditedAmount,
                'benefit_description' => $this->formatBenefitDescription($offer, $rechargeAmount, $discountAmount, $cashbackAmount),
            ];
        }

        return response()->json([
            'success' => true,
            'message' => 'Top-up request submitted successfully. Waiting for admin approval.',
            'data' => $responseData
        ], 201);
    }

    /**
     * Format benefit description for response
     */
    private function formatBenefitDescription($offer, $rechargeAmount, $discountAmount, $cashbackAmount)
    {
        $parts = [];
        
        if ($discountAmount > 0) {
            if ($offer->discount_type === 'percentage') {
                $parts[] = "{$offer->discount_value}% discount = ₹{$discountAmount}";
            } else {
                $parts[] = "₹{$discountAmount} discount";
            }
        }
        
        if ($cashbackAmount > 0) {
            $parts[] = "₹{$cashbackAmount} cashback";
        }
        
        return implode(' + ', $parts) ?: 'No benefits';
    }

    /**
     * Get User's Top-Up Requests
     */
    public function index(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $requests = TopUpRequest::where('user_id', $request->user_id)
            ->with(['upi', 'admin', 'rechargeOffer'])
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $requests
        ], 200);
    }

    /**
     * Get Single Top-Up Request
     */
    public function show($id)
    {
        $request = TopUpRequest::with(['user', 'upi', 'admin', 'rechargeOffer'])->find($id);

        if (!$request) {
            return response()->json([
                'success' => false,
                'message' => 'Top-up request not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $request
        ], 200);
    }

    /**
     * Get Wallet Top-Up History
     * Returns formatted history of all top-up requests for a user
     */
    public function history(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'status' => 'nullable|in:pending,approved,rejected',
            'limit' => 'nullable|integer|min:1|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $query = TopUpRequest::where('user_id', $request->user_id)
            ->with(['upi', 'admin', 'rechargeOffer']);

        // Filter by status if provided
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Apply limit (default: 50)
        $limit = $request->input('limit', 50);
        $requests = $query->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();

        // Format response with summary
        $summary = [
            'total_requests' => TopUpRequest::where('user_id', $request->user_id)->count(),
            'pending' => TopUpRequest::where('user_id', $request->user_id)->where('status', 'pending')->count(),
            'approved' => TopUpRequest::where('user_id', $request->user_id)->where('status', 'approved')->count(),
            'rejected' => TopUpRequest::where('user_id', $request->user_id)->where('status', 'rejected')->count(),
            'total_approved_amount' => TopUpRequest::where('user_id', $request->user_id)
                ->where('status', 'approved')
                ->sum('amount'),
        ];

        return response()->json([
            'success' => true,
            'summary' => $summary,
            'data' => $requests
        ], 200);
    }
}
