<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\ReferralRelationship;
use App\Models\ReferralEarning;
use App\Services\ReferralService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ReferralController extends Controller
{
    /**
     * Get User's Referral Code
     */
    public function getReferralCode(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::find($request->user_id);

        // Generate referral code if doesn't exist
        if (!$user->referral_code) {
            $user->referral_code = User::generateReferralCode();
            $user->save();
        }

        return response()->json([
            'success' => true,
            'data' => [
                'referral_code' => $user->referral_code,
                'referral_link' => url('/signup?ref=' . $user->referral_code),
            ]
        ], 200);
    }

    /**
     * Get User's Downline (People they referred)
     */
    public function getDownline(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'level' => 'nullable|integer|min:1|max:10',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::find($request->user_id);
        $level = $request->input('level', 1);

        $query = ReferralRelationship::where('referrer_id', $user->id)
            ->where('level', $level)
            ->with(['referred:id,name,email,phone_number,referral_code,created_at']);

        $relationships = $query->get();

        $downline = $relationships->map(function ($rel) {
            return [
                'id' => $rel->referred->id,
                'name' => $rel->referred->name,
                'email' => $rel->referred->email,
                'phone_number' => $rel->referred->phone_number,
                'referral_code' => $rel->referred->referral_code,
                'joined_at' => $rel->referred->created_at,
                'level' => $rel->level,
            ];
        });

        // Get summary
        $summary = [
            'total_level_1' => ReferralRelationship::where('referrer_id', $user->id)->where('level', 1)->count(),
            'total_level_2' => ReferralRelationship::where('referrer_id', $user->id)->where('level', 2)->count(),
            'total_level_3' => ReferralRelationship::where('referrer_id', $user->id)->where('level', 3)->count(),
        ];

        return response()->json([
            'success' => true,
            'level' => $level,
            'summary' => $summary,
            'data' => $downline
        ], 200);
    }

    /**
     * Get User's Referral Earnings
     */
    public function getEarnings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'status' => 'nullable|in:pending,credited,cancelled',
            'earning_type' => 'nullable|in:signup,topup,transaction',
            'limit' => 'nullable|integer|min:1|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::find($request->user_id);
        $limit = $request->input('limit', 50);

        $query = ReferralEarning::where('referrer_id', $user->id)
            ->with(['referred:id,name,email,phone_number']);

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('earning_type')) {
            $query->where('earning_type', $request->earning_type);
        }

        $earnings = $query->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();

        // Get summary
        $summary = [
            'total_earnings' => ReferralEarning::where('referrer_id', $user->id)
                ->where('status', 'credited')
                ->sum('commission_amount'),
            'pending_earnings' => ReferralEarning::where('referrer_id', $user->id)
                ->where('status', 'pending')
                ->sum('commission_amount'),
            'total_count' => ReferralEarning::where('referrer_id', $user->id)->count(),
            'credited_count' => ReferralEarning::where('referrer_id', $user->id)
                ->where('status', 'credited')
                ->count(),
        ];

        return response()->json([
            'success' => true,
            'summary' => $summary,
            'data' => $earnings
        ], 200);
    }

    /**
     * Credit Pending Earnings to Wallet
     */
    public function creditEarnings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'earning_ids' => 'nullable|array',
            'earning_ids.*' => 'exists:referral_earnings,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::find($request->user_id);
        $earningIds = $request->input('earning_ids');

        $creditedAmount = ReferralService::creditEarnings($user, $earningIds);

        if ($creditedAmount > 0) {
            return response()->json([
                'success' => true,
                'message' => "₹{$creditedAmount} credited to wallet successfully",
                'data' => [
                    'credited_amount' => $creditedAmount,
                    'new_wallet_balance' => $user->fresh()->wallet_balance,
                ]
            ], 200);
        }

        return response()->json([
            'success' => false,
            'message' => 'No pending earnings to credit'
        ], 400);
    }
}
