<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\TopUpRequest;
use App\Models\User;
use App\Models\Upi;
use App\Services\ReferralService;
use App\Events\PaymentStatusUpdated;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    /**
     * Check Payment Status
     * After submitting a top-up request, UI can poll this endpoint to check status
     */
    public function checkStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'request_id' => 'required|exists:top_up_requests,id',
            'user_id' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $topUpRequest = TopUpRequest::with(['user', 'upi', 'rechargeOffer'])
            ->where('id', $request->request_id)
            ->where('user_id', $request->user_id)
            ->first();

        if (!$topUpRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Top-up request not found or does not belong to this user'
            ], 404);
        }

        $response = [
            'success' => true,
            'data' => [
                'request_id' => $topUpRequest->id,
                'status' => $topUpRequest->status,
                'amount' => $topUpRequest->amount,
                'final_credited_amount' => $topUpRequest->final_credited_amount,
                'upi_details' => [
                    'upi_id' => $topUpRequest->upi->upi_id,
                    'name' => $topUpRequest->upi->name,
                ],
                'created_at' => $topUpRequest->created_at,
                'updated_at' => $topUpRequest->updated_at,
            ]
        ];

        // Add status-specific messages
        switch ($topUpRequest->status) {
            case 'verifying_payment':
                $response['data']['message'] = 'Please wait, we are checking the payment status...';
                break;
            case 'approved':
                $response['data']['message'] = 'Payment verified and approved successfully!';
                $response['data']['user_new_balance'] = $topUpRequest->user->wallet_balance;
                break;
            case 'rejected':
                $response['data']['message'] = 'Payment request was rejected.';
                $response['data']['rejection_reason'] = $topUpRequest->rejection_reason;
                break;
            case 'pending':
                $response['data']['message'] = 'Payment request is pending admin approval.';
                break;
        }

        return response()->json($response, 200);
    }

    /**
     * Process SMS from Mobile App
     * Mobile app sends SMS content, we parse it and match with pending requests
     */
    public function processSms(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sms_content' => 'required|string',
            'user_id' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $smsContent = $request->sms_content;
        $userId = $request->user_id;

        Log::info('SMS received for processing', [
            'user_id' => $userId,
            'sms_length' => strlen($smsContent)
        ]);

        // Parse SMS to extract amount and UPI ID
        $parsedData = $this->parseSms($smsContent);

        if (!$parsedData) {
            return response()->json([
                'success' => false,
                'message' => 'Could not parse payment information from SMS'
            ], 400);
        }

        Log::info('SMS parsed successfully', [
            'user_id' => $userId,
            'parsed_amount' => $parsedData['amount'],
            'parsed_upi' => $parsedData['upi_id'] ?? 'not found'
        ]);

        // Find pending/verifying payment requests for this user
        $topUpRequest = TopUpRequest::where('user_id', $userId)
            ->whereIn('status', ['pending', 'verifying_payment'])
            ->orderBy('created_at', 'desc')
            ->first();

        if (!$topUpRequest) {
            return response()->json([
                'success' => false,
                'message' => 'No pending payment request found for this user'
            ], 404);
        }

        // Get UPI details
        $upi = $topUpRequest->upi;

        // Check if amount matches (with small tolerance for rounding)
        $amountMatch = abs($parsedData['amount'] - $topUpRequest->amount) < 0.01;

        // Check if UPI matches (if UPI was found in SMS)
        $upiMatch = true;
        if (isset($parsedData['upi_id']) && !empty($parsedData['upi_id'])) {
            $upiMatch = stripos($upi->upi_id, $parsedData['upi_id']) !== false || 
                       stripos($parsedData['upi_id'], $upi->upi_id) !== false;
        }

        Log::info('Payment verification check', [
            'request_id' => $topUpRequest->id,
            'expected_amount' => $topUpRequest->amount,
            'received_amount' => $parsedData['amount'],
            'amount_match' => $amountMatch,
            'expected_upi' => $upi->upi_id,
            'received_upi' => $parsedData['upi_id'] ?? 'not found',
            'upi_match' => $upiMatch
        ]);

        if ($amountMatch && $upiMatch) {
            // Auto-approve the payment
            $user = $topUpRequest->user;
            $user->wallet_balance += $topUpRequest->final_credited_amount;
            $user->save();

            // Process referral earnings
            ReferralService::processTransactionReferral(
                $user,
                $topUpRequest->final_credited_amount,
                $topUpRequest->id,
                'TopUpRequest'
            );

            // Update request status
            $topUpRequest->update([
                'status' => 'approved',
                'approved_at' => now(),
            ]);

            // Broadcast payment status update
            event(new PaymentStatusUpdated($topUpRequest, 'approved'));

            Log::info('Payment auto-approved via SMS', [
                'request_id' => $topUpRequest->id,
                'user_id' => $userId,
                'amount' => $topUpRequest->amount,
                'credited_amount' => $topUpRequest->final_credited_amount
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment verified and approved successfully!',
                'data' => [
                    'request_id' => $topUpRequest->id,
                    'status' => 'approved',
                    'amount' => $topUpRequest->amount,
                    'final_credited_amount' => $topUpRequest->final_credited_amount,
                    'user_new_balance' => $user->wallet_balance,
                ]
            ], 200);
        } else {
            // Amount or UPI doesn't match
            $reason = [];
            if (!$amountMatch) {
                $reason[] = "Amount mismatch: Expected ₹{$topUpRequest->amount}, Received ₹{$parsedData['amount']}";
            }
            if (!$upiMatch) {
                $reason[] = "UPI mismatch";
            }

            Log::warning('Payment verification failed', [
                'request_id' => $topUpRequest->id,
                'reasons' => $reason
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Payment verification failed',
                'data' => [
                    'request_id' => $topUpRequest->id,
                    'status' => $topUpRequest->status,
                    'reasons' => $reason,
                    'expected_amount' => $topUpRequest->amount,
                    'received_amount' => $parsedData['amount'],
                ]
            ], 400);
        }
    }

    /**
     * Parse SMS content to extract payment information
     * Handles various UPI payment SMS formats
     */
    private function parseSms($smsContent)
    {
        $result = [
            'amount' => null,
            'upi_id' => null,
        ];

        // Common UPI payment SMS patterns
        // Pattern 1: "Rs. 100.00 credited to account" or "₹100.00 credited"
        // Pattern 2: "INR 100.00 received from" or "Received INR 100.00"
        // Pattern 3: "Amount: ₹100.00" or "Amount: Rs. 100.00"
        
        // Extract amount - try multiple patterns
        $amountPatterns = [
            '/[Rr][Ss]\.?\s*(\d+(?:\.\d{2})?)/',           // Rs. 100.00 or Rs 100
            '/₹\s*(\d+(?:\.\d{2})?)/',                     // ₹100.00
            '/INR\s*(\d+(?:\.\d{2})?)/',                   // INR 100.00
            '/amount[:\s]+[Rr][Ss]\.?\s*(\d+(?:\.\d{2})?)/i', // Amount: Rs. 100.00
            '/amount[:\s]+₹\s*(\d+(?:\.\d{2})?)/i',       // Amount: ₹100.00
            '/amount[:\s]+INR\s*(\d+(?:\.\d{2})?)/i',      // Amount: INR 100.00
            '/credited[:\s]+[Rr][Ss]\.?\s*(\d+(?:\.\d{2})?)/i', // Credited Rs. 100.00
            '/received[:\s]+INR\s*(\d+(?:\.\d{2})?)/i',    // Received INR 100.00
            '/(\d+(?:\.\d{2})?)\s*credited/i',            // 100.00 credited
            '/(\d+(?:\.\d{2})?)\s*received/i',            // 100.00 received
        ];

        foreach ($amountPatterns as $pattern) {
            if (preg_match($pattern, $smsContent, $matches)) {
                $result['amount'] = (float) $matches[1];
                break;
            }
        }

        // Extract UPI ID - common patterns
        $upiPatterns = [
            '/([a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/',  // email format UPI
            '/([a-zA-Z0-9._-]+@paytm)/i',                        // paytm UPI
            '/([a-zA-Z0-9._-]+@ybl)/i',                          // phonepe UPI
            '/([a-zA-Z0-9._-]+@okaxis)/i',                       // axis bank UPI
            '/([a-zA-Z0-9._-]+@okhdfcbank)/i',                  // HDFC bank UPI
            '/([a-zA-Z0-9._-]+@okicici)/i',                      // ICICI bank UPI
        ];

        foreach ($upiPatterns as $pattern) {
            if (preg_match($pattern, $smsContent, $matches)) {
                $result['upi_id'] = strtolower($matches[1]);
                break;
            }
        }

        // If amount was found, return result (UPI is optional)
        if ($result['amount'] !== null) {
            return $result;
        }

        return null;
    }
}

