<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Gift;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;

class GiftController extends Controller
{
    /**
     * Get authenticated user from token
     */
    private function getUser(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $user = $accessToken->tokenable;
        if (!$user || !($user instanceof User)) {
            return null;
        }

        return $user;
    }

    /**
     * Get Unread Gifts
     * Returns gifts that user hasn't seen yet (for popup display)
     */
    public function getUnreadGifts(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $gifts = Gift::where('user_id', $user->id)
            ->where('is_read', false)
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'count' => $gifts->count(),
            'data' => $gifts->map(function ($gift) {
                return [
                    'id' => $gift->id,
                    'amount' => (float) $gift->amount,
                    'message' => $gift->message,
                    'created_at' => $gift->created_at,
                ];
            })
        ], 200);
    }

    /**
     * Mark Gift as Read
     * User marks a gift as read (after viewing popup)
     */
    public function markAsRead(Request $request, $giftId)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $gift = Gift::where('id', $giftId)
            ->where('user_id', $user->id)
            ->first();

        if (!$gift) {
            return response()->json([
                'success' => false,
                'message' => 'Gift not found'
            ], 404);
        }

        if ($gift->is_read) {
            return response()->json([
                'success' => false,
                'message' => 'Gift already marked as read'
            ], 400);
        }

        $gift->update([
            'is_read' => true,
            'read_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Gift marked as read',
            'data' => [
                'id' => $gift->id,
                'is_read' => true,
                'read_at' => $gift->read_at,
            ]
        ], 200);
    }

    /**
     * Mark All Gifts as Read
     * User marks all unread gifts as read
     */
    public function markAllAsRead(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $count = Gift::where('user_id', $user->id)
            ->where('is_read', false)
            ->update([
                'is_read' => true,
                'read_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => "{$count} gift(s) marked as read",
            'data' => [
                'marked_count' => $count,
            ]
        ], 200);
    }

    /**
     * Get All Gifts (User's Gift History)
     */
    public function index(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $query = Gift::where('user_id', $user->id);

        // Filter by is_read
        if ($request->has('is_read')) {
            $query->where('is_read', $request->is_read === 'true' || $request->is_read === '1');
        }

        // Sort options
        $sortBy = $request->input('sort_by', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->input('per_page', 15);
        $perPage = min(max($perPage, 1), 100);

        $gifts = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $gifts->items(),
            'pagination' => [
                'current_page' => $gifts->currentPage(),
                'per_page' => $gifts->perPage(),
                'total' => $gifts->total(),
                'last_page' => $gifts->lastPage(),
            ]
        ], 200);
    }
}

