<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Service;
use App\Models\ServiceTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\PersonalAccessToken;
use Carbon\Carbon;

class BocwIdController extends Controller
{
    /**
     * Get authenticated user from token
     */
    private function getUser(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $user = $accessToken->tokenable;
        if (!$user || !($user instanceof User)) {
            return null;
        }

        return $user;
    }

    /**
     * Format date to dd/mm/yyyy
     */
    private function formatDate($dateString)
    {
        if (!$dateString) {
            return null;
        }
        
        try {
            $date = Carbon::parse($dateString);
            return $date->format('d/m/Y');
        } catch (\Exception $e) {
            return $dateString;
        }
    }

    /**
     * Get gender in Marathi
     */
    private function getGenderMarathi($genderCode)
    {
        // genderPersonal = 3 means female
        if ($genderCode == 3) {
            return 'स्त्री';
        } elseif ($genderCode == 1 || $genderCode == 2) {
            return 'पुरुष';
        }
        return null;
    }

    /**
     * Verify BOCW ID using Aadhar Number
     */
    public function verify(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'aadhar' => 'required|string|size:12|regex:/^[0-9]{12}$/',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Get BOCW ID service
        $service = Service::where('path', '/user/bocw')->first();
        
        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'BOCW ID service not found. Please contact admin.'
            ], 404);
        }

        if ($service->is_coming_soon) {
            return response()->json([
                'success' => false,
                'message' => 'This service is coming soon.'
            ], 400);
        }

        $serviceCost = $service->cost ?? 25.00; // Default to ₹25 if not set

        // Check wallet balance
        if ($user->wallet_balance < $serviceCost) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient wallet balance',
                'data' => [
                    'required_amount' => $serviceCost,
                    'current_balance' => $user->wallet_balance,
                    'shortage' => $serviceCost - $user->wallet_balance
                ]
            ], 400);
        }

        // Create transaction record
        $transaction = ServiceTransaction::create([
            'user_id' => $user->id,
            'service_id' => $service->id,
            'amount' => $serviceCost,
            'service_type' => 'bocw_id',
            'request_data' => ['aadhar' => $request->aadhar],
            'status' => 'pending',
        ]);

        try {
            // Call external API to get BOCW details
            // The API endpoint: https://iwbms.mahabocw.in/api/bocw-registration/getsearchadDetail
            // Try different methods and parameter formats
            
            $apiUrl = 'https://iwbms.mahabocw.in/api/bocw-registration/getsearchadDetail';
            $response = null;
            $responseData = null;
            
            // Try 1: POST with JSON body
            try {
                $response = Http::timeout(180)
                    ->withHeaders([
                        'Content-Type' => 'application/json',
                        'Accept' => 'application/json'
                    ])
                    ->post($apiUrl, [
                        'aadhar' => $request->aadhar
                    ]);
                    
                if ($response->successful()) {
                    $responseData = $response->json();
                }
            } catch (\Exception $e) {
                \Log::warning('BOCW API POST failed: ' . $e->getMessage());
            }
            
            // Try 2: GET with query parameter (if POST failed or returned 404)
            if (!$response || $response->status() === 404 || !$responseData) {
                try {
                    $response = Http::timeout(180)
                        ->withHeaders([
                            'Accept' => 'application/json'
                        ])
                        ->get($apiUrl, [
                            'aadhar' => $request->aadhar
                        ]);
                        
                    $responseData = $response->json();
                } catch (\Exception $e) {
                    \Log::warning('BOCW API GET failed: ' . $e->getMessage());
                }
            }
            
            // Log the response for debugging
            if ($response) {
                \Log::info('BOCW API Response', [
                    'method' => $response->transferStats ? 'POST' : 'GET',
                    'status' => $response->status(),
                    'url' => $apiUrl,
                    'body' => $responseData
                ]);
            }

            // Check if we have a valid response
            if (!$response) {
                throw new \Exception('Failed to connect to BOCW API. Please try again later.');
            }
            
            // Check if API call was successful
            if ($response->successful() && is_array($responseData) && count($responseData) > 0) {
                $bocwData = $responseData[0]; // Get first record
                $bocwId = $bocwData['bocw_id'] ?? null;

                if (!$bocwId) {
                    throw new \Exception('BOCW ID not found in response');
                }

                // Get photo filename
                $photoFilename = null;
                $photoUrl = null;
                try {
                    $photoResponse = Http::timeout(30)->post('https://iwbms.mahabocw.in/api/bocw-registration/getregistrationbybocwidforprofile', [
                        'bocwId' => $bocwId,
                        'panelId' => 'applicant'
                    ]);

                    if ($photoResponse->successful()) {
                        $photoData = $photoResponse->json();
                        $photoFilename = $photoData['applicantPhotoFile'] ?? null;
                        
                        if ($photoFilename) {
                            $photoUrl = 'https://storage.mahabocw.in/getfile/' . $photoFilename;
                        }
                    }
                } catch (\Exception $e) {
                    // Photo fetch failed, but continue with other data
                    \Log::warning('Failed to fetch BOCW photo: ' . $e->getMessage());
                }

                // Format the response data
                $formattedData = [
                    'registration_no' => $bocwData['registration_no'] ?? null,
                    'name_marathi' => trim(($bocwData['firstNamePersonal_mr'] ?? '') . ' ' . ($bocwData['middleNamePersonal_mr'] ?? '') . ' ' . ($bocwData['lastNamePersonal_mr'] ?? '')),
                    'gender_marathi' => $this->getGenderMarathi($bocwData['genderPersonal'] ?? null),
                    'dob' => $this->formatDate($bocwData['dobPersonal'] ?? null),
                    'mobile' => $bocwData['mobilePersonal'] ?? null,
                    'residential_city_marathi' => $bocwData['residential_address_city_mr'] ?? null,
                    'permanent_district_marathi' => $bocwData['permanent_address_district_mr'] ?? null,
                    'registration_date' => $this->formatDate($bocwData['registrationDatePersonal'] ?? null),
                    'photo_url' => $photoUrl,
                ];

                // Deduct amount from wallet
                DB::transaction(function () use ($user, $serviceCost, $transaction, $responseData, $formattedData) {
                    $user->wallet_balance -= $serviceCost;
                    $user->save();

                    // Update transaction
                    $transaction->update([
                        'status' => 'success',
                        'response_data' => [
                            'raw_data' => $responseData,
                            'formatted_data' => $formattedData
                        ],
                    ]);
                });

                // Return formatted data
                return response()->json([
                    'success' => true,
                    'message' => 'BOCW ID verified successfully',
                    'data' => array_merge($formattedData, [
                        'transaction_id' => $transaction->id,
                        'amount_deducted' => $serviceCost,
                        'remaining_balance' => $user->fresh()->wallet_balance,
                    ])
                ], 200);
            } else {
                // API call failed or returned error
                $errorMessage = 'No BOCW registration found for this Aadhar number';
                
                // Handle different error scenarios
                if ($response->status() === 404) {
                    $errorMessage = 'BOCW API endpoint not found. Please contact support.';
                } elseif (is_array($responseData) && count($responseData) === 0) {
                    $errorMessage = 'No BOCW registration found for this Aadhar number';
                } elseif (isset($responseData['message'])) {
                    $errorMessage = $responseData['message'];
                } elseif (isset($responseData['error'])) {
                    $errorMessage = $responseData['error'];
                }

                $transaction->update([
                    'status' => 'failed',
                    'response_data' => $responseData,
                    'error_message' => $errorMessage,
                ]);

                return response()->json([
                    'success' => false,
                    'message' => $errorMessage,
                    'data' => $responseData,
                    'status_code' => $response->status()
                ], $response->status() === 404 ? 404 : 400);
            }
        } catch (\Exception $e) {
            // Handle exceptions
            $transaction->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while verifying BOCW ID. Please try again later.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get Transaction Status by ID
     */
    public function getStatus(Request $request, $id)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $transaction = ServiceTransaction::where('id', $id)
            ->where('user_id', $user->id)
            ->with(['service:id,name,path'])
            ->first();

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction not found'
            ], 404);
        }

        $responseData = [
            'id' => $transaction->id,
            'status' => $transaction->status,
            'amount' => $transaction->amount,
            'service' => $transaction->service,
            'created_at' => $transaction->created_at,
            'updated_at' => $transaction->updated_at,
        ];

        // If successful, include BOCW data
        if ($transaction->status === 'success' && $transaction->response_data) {
            $formattedData = $transaction->response_data['formatted_data'] ?? null;
            if ($formattedData) {
                $responseData['bocw_data'] = $formattedData;
            }
        }

        // If failed, include error message
        if ($transaction->status === 'failed') {
            $responseData['error_message'] = $transaction->error_message;
        }

        return response()->json([
            'success' => true,
            'data' => $responseData
        ], 200);
    }

    /**
     * Get Service Transaction History
     */
    public function history(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $transactions = ServiceTransaction::where('user_id', $user->id)
            ->where('service_type', 'bocw_id')
            ->with(['service:id,name,path'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $transactions
        ], 200);
    }
}
