<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\TopUpRequest;
use App\Models\Admin;
use App\Services\ReferralService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;

class TopUpRequestController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Get All Top-Up Requests
     */
    public function index(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $status = $request->query('status', 'all');
        
        $query = TopUpRequest::with(['user', 'upi', 'admin', 'rechargeOffer']);
        
        if ($status !== 'all') {
            $query->where('status', $status);
        }

        $requests = $query->orderBy('created_at', 'desc')->get();

        return response()->json([
            'success' => true,
            'data' => $requests
        ], 200);
    }

    /**
     * Get Single Top-Up Request
     */
    public function show(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $topUpRequest = TopUpRequest::with(['user', 'upi', 'admin', 'rechargeOffer'])->find($id);

        if (!$topUpRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Top-up request not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $topUpRequest
        ], 200);
    }

    /**
     * Approve Top-Up Request
     */
    public function approve(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $topUpRequest = TopUpRequest::with(['user', 'upi', 'rechargeOffer'])->find($id);

        if (!$topUpRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Top-up request not found'
            ], 404);
        }

        if ($topUpRequest->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Top-up request is not pending. Current status: ' . $topUpRequest->status
            ], 400);
        }

        // Update wallet balance with final credited amount (includes discount + cashback)
        $user = $topUpRequest->user;
        $user->wallet_balance += $topUpRequest->final_credited_amount;
        $user->save();

        // Process referral earnings for top-up
        ReferralService::processTransactionReferral(
            $user,
            $topUpRequest->final_credited_amount,
            $topUpRequest->id,
            'TopUpRequest'
        );

        // Update request status
        $topUpRequest->update([
            'status' => 'approved',
            'approved_by' => $admin->id,
            'approved_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Top-up request approved successfully',
            'data' => [
                'request' => $topUpRequest->fresh(['user', 'upi', 'admin', 'rechargeOffer']),
                'user_new_balance' => $user->wallet_balance,
                'credited_amount' => $topUpRequest->final_credited_amount,
                'offer_benefits' => $topUpRequest->rechargeOffer ? [
                    'discount' => $topUpRequest->discount_amount,
                    'cashback' => $topUpRequest->cashback_amount,
                ] : null,
            ]
        ], 200);
    }

    /**
     * Reject Top-Up Request
     */
    public function reject(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'rejection_reason' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $topUpRequest = TopUpRequest::find($id);

        if (!$topUpRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Top-up request not found'
            ], 404);
        }

        if ($topUpRequest->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Top-up request is not pending. Current status: ' . $topUpRequest->status
            ], 400);
        }

        $topUpRequest->update([
            'status' => 'rejected',
            'rejection_reason' => $request->rejection_reason,
            'approved_by' => $admin->id,
            'approved_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Top-up request rejected successfully',
            'data' => $topUpRequest->fresh(['user', 'upi', 'admin', 'rechargeOffer'])
        ], 200);
    }
}
