<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\StateUrl;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class StateUrlController extends Controller
{
    /**
     * List all State URLs
     */
    public function index()
    {
        $stateUrls = StateUrl::all();
        return response()->json([
            'success' => true,
            'data' => $stateUrls
        ], 200);
    }

    /**
     * Create new State URL
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'state_name' => 'required|string|unique:state_urls,state_name',
            'url' => 'required|url',
            'api_url' => 'nullable|url',
            'includes_pattern' => 'required|string',
            'token' => 'nullable|string',
            'username' => 'nullable|string',
            'password' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $stateUrl = StateUrl::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'State URL created successfully',
            'data' => $stateUrl
        ], 201);
    }

    /**
     * Get single State URL
     */
    public function show($id)
    {
        $stateUrl = StateUrl::find($id);

        if (!$stateUrl) {
            return response()->json([
                'success' => false,
                'message' => 'State URL not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $stateUrl
        ], 200);
    }

    /**
     * Update State URL
     */
    public function update(Request $request, $id)
    {
        $stateUrl = StateUrl::find($id);

        if (!$stateUrl) {
            return response()->json([
                'success' => false,
                'message' => 'State URL not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'state_name' => 'sometimes|required|string|unique:state_urls,state_name,' . $id,
            'url' => 'sometimes|required|url',
            'api_url' => 'nullable|url',
            'includes_pattern' => 'sometimes|required|string',
            'token' => 'nullable|string',
            'username' => 'nullable|string',
            'password' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $stateUrl->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'State URL updated successfully',
            'data' => $stateUrl
        ], 200);
    }


    /**
     * Delete State URL
     */
    public function destroy($id)
    {
        $stateUrl = StateUrl::find($id);

        if (!$stateUrl) {
            return response()->json([
                'success' => false,
                'message' => 'State URL not found'
            ], 404);
        }

        $stateUrl->delete();

        return response()->json([
            'success' => true,
            'message' => 'State URL deleted successfully'
        ], 200);
    }

    /**
     * Fetch Config by State Name
     */
    public function fetchConfig(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'state_name' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $stateUrl = StateUrl::where('state_name', $request->state_name)->first();

        if (!$stateUrl) {
            return response()->json([
                'success' => false,
                'message' => 'Configuration not found for the specified state'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $stateUrl
        ], 200);
    }

    /**
     * Update Token (and Credentials) by State Name
     */
    public function updateToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'state_name' => 'required|string',
            'token' => 'nullable|string',
            'username' => 'nullable|string',
            'password' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $stateUrl = StateUrl::where('state_name', $request->state_name)->first();

        if (!$stateUrl) {
            return response()->json([
                'success' => false,
                'message' => 'Configuration not found for the specified state'
            ], 404);
        }

        $updateData = [];
        if ($request->has('token')) $updateData['token'] = $request->token;
        if ($request->has('username')) $updateData['username'] = $request->username;
        if ($request->has('password')) $updateData['password'] = $request->password;
        if ($request->has('api_url')) $updateData['api_url'] = $request->api_url;

        $stateUrl->update($updateData);

        return response()->json([
            'success' => true,
            'message' => 'Token/Credentials updated successfully',
            'data' => $stateUrl
        ], 200);
    }
}
