<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Service;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\PersonalAccessToken;
use Symfony\Component\HttpFoundation\File\UploadedFile;

class ServiceController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Validate manually parsed file
     */
    private function validateManuallyParsedFile($file)
    {
        $errors = [];
        
        // Check if file exists
        if (!file_exists($file->getPathname())) {
            $errors[] = 'The image file does not exist.';
            return $errors;
        }
        
        // Check file size (max 2048 KB = 2MB)
        $fileSize = filesize($file->getPathname());
        $maxSize = 2048 * 1024; // 2MB in bytes
        if ($fileSize > $maxSize) {
            $errors[] = 'The image must not be greater than 2048 kilobytes.';
        }
        
        // Check if it's a valid image
        $imageInfo = @getimagesize($file->getPathname());
        if ($imageInfo === false) {
            $errors[] = 'The image must be an image.';
            return $errors;
        }
        
        // Check MIME type
        $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
        $mimeType = $imageInfo['mime'] ?? $file->getMimeType();
        if (!in_array($mimeType, $allowedMimes)) {
            $errors[] = 'The image must be a file of type: jpeg, png, jpg, gif, webp.';
        }
        
        // Check file extension
        $extension = strtolower($file->getClientOriginalExtension());
        $allowedExtensions = ['jpeg', 'png', 'jpg', 'gif', 'webp'];
        if (!in_array($extension, $allowedExtensions)) {
            $errors[] = 'The image must be a file of type: jpeg, png, jpg, gif, webp.';
        }
        
        return $errors;
    }

    /**
     * Parse file from PUT request with multipart/form-data
     * Laravel doesn't automatically parse files in PUT requests, so we need to do it manually
     */
    private function parseFileFromPutRequest(Request $request, $fieldName)
    {
        try {
            $contentType = $request->header('Content-Type', '');
            
            if (!str_contains($contentType, 'multipart/form-data')) {
                return null;
            }

            // Extract boundary from Content-Type header
            preg_match('/boundary=(.*)$/i', $contentType, $matches);
            if (!isset($matches[1])) {
                return null;
            }
            
            $boundary = '--' . trim($matches[1]);
            
            // For PUT requests, read from php://input directly
            $rawContent = file_get_contents('php://input');
            
            if (empty($rawContent)) {
                \Log::warning('PUT request has empty content');
                return null;
            }
            
            \Log::info('Parsing PUT request content', [
                'contentLength' => strlen($rawContent),
                'boundary' => $boundary,
                'hasBoundary' => strpos($rawContent, $boundary) !== false
            ]);

            // Split by boundary
            $parts = explode($boundary, $rawContent);
            
            foreach ($parts as $part) {
                // Skip empty parts
                if (trim($part) === '' || trim($part) === '--') {
                    continue;
                }

                // Check if this part contains the file field
                if (strpos($part, 'name="' . $fieldName . '"') === false) {
                    continue;
                }

                // Extract headers and content
                list($headers, $content) = explode("\r\n\r\n", $part, 2);
                
                // Extract filename from headers
                if (preg_match('/filename="([^"]+)"/', $headers, $filenameMatches)) {
                    $filename = $filenameMatches[1];
                } elseif (preg_match("/filename='([^']+)'/", $headers, $filenameMatches)) {
                    $filename = $filenameMatches[1];
                } else {
                    continue; // No filename found, skip
                }

                // Extract content type
                $mimeType = 'application/octet-stream';
                if (preg_match('/Content-Type:\s*([^\r\n]+)/i', $headers, $mimeMatches)) {
                    $mimeType = trim($mimeMatches[1]);
                }

                // Remove trailing boundary markers and whitespace
                $content = rtrim($content, "\r\n--");

                // Create temporary file
                $tempPath = tempnam(sys_get_temp_dir(), 'laravel_upload_');
                file_put_contents($tempPath, $content);

                // Get file extension from original filename
                $extension = pathinfo($filename, PATHINFO_EXTENSION);
                if (empty($extension)) {
                    // Try to get extension from mime type
                    $extensions = [
                        'image/jpeg' => 'jpg',
                        'image/png' => 'png',
                        'image/gif' => 'gif',
                        'image/webp' => 'webp',
                    ];
                    $extension = $extensions[$mimeType] ?? 'bin';
                }

                // Create UploadedFile instance
                // Parameters: path, originalName, mimeType, error, test
                $uploadedFile = new UploadedFile(
                    $tempPath,
                    $filename,
                    $mimeType,
                    UPLOAD_ERR_OK,
                    false // test mode = false (real upload)
                );

                \Log::info('File parsed from PUT request', [
                    'field' => $fieldName,
                    'filename' => $filename,
                    'size' => strlen($content),
                    'mimeType' => $mimeType,
                    'tempPath' => $tempPath
                ]);

                return $uploadedFile;
            }

            return null;
        } catch (\Exception $e) {
            \Log::error('Error parsing file from PUT request', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * Get All Services
     */
    public function index(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $services = Service::orderBy('created_at', 'desc')->get();

        // Format services with image URLs
        $servicesData = $services->map(function ($service) {
            $serviceData = $service->toArray();
            if ($service->image) {
                $serviceData['image_url'] = Storage::url($service->image);
            }
            return $serviceData;
        });

        return response()->json([
            'success' => true,
            'data' => $servicesData
        ], 200);
    }

    /**
     * Get Single Service
     */
    public function show(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $service = Service::find($id);

        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Service not found'
            ], 404);
        }

        // Format service with image URL
        $serviceData = $service->toArray();
        if ($service->image) {
            $serviceData['image_url'] = Storage::url($service->image);
        }

        return response()->json([
            'success' => true,
            'data' => $serviceData
        ], 200);
    }

    /**
     * Add New Service
     */
    public function store(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'path' => 'required|string|unique:services,path|regex:/^\/[a-zA-Z0-9\/-]+$/',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'is_coming_soon' => 'nullable|boolean',
            'cost' => 'nullable|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $imagePath = null;
        
        // Handle image upload
        if ($request->hasFile('image')) {
            try {
                $image = $request->file('image');
                
                // Debug logging
                \Log::info('Image upload attempt', [
                    'hasFile' => $request->hasFile('image'),
                    'isValid' => $image->isValid(),
                    'originalName' => $image->getClientOriginalName(),
                    'size' => $image->getSize(),
                    'mimeType' => $image->getMimeType()
                ]);
                
                if (!$image->isValid()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid image file'
                    ], 422);
                }
                
                // Ensure the directory exists
                $directory = storage_path('app/public/services');
                if (!file_exists($directory)) {
                    mkdir($directory, 0755, true);
                }
                
                $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                $imagePath = $image->storeAs('services', $imageName, 'public');
                
                \Log::info('Image stored', [
                    'imagePath' => $imagePath,
                    'fullPath' => storage_path('app/public/' . $imagePath),
                    'exists' => Storage::disk('public')->exists($imagePath)
                ]);
                
                // Verify file was saved
                if (!$imagePath || !Storage::disk('public')->exists($imagePath)) {
                    \Log::error('Image upload failed', [
                        'imagePath' => $imagePath,
                        'directory' => $directory,
                        'exists' => $imagePath ? Storage::disk('public')->exists($imagePath) : false
                    ]);
                    
                    return response()->json([
                        'success' => false,
                        'message' => 'Image file was not saved successfully'
                    ], 422);
                }
            } catch (\Exception $e) {
                \Log::error('Image upload exception', [
                    'message' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Image upload failed: ' . $e->getMessage()
                ], 422);
            }
        }

        $service = Service::create([
            'name' => $request->name,
            'path' => $request->path,
            'image' => $imagePath,
            'is_coming_soon' => $request->is_coming_soon ?? false,
            'cost' => $request->cost ?? 0.00,
            'created_by' => $admin->id,
        ]);

        // Format response with image URL
        $serviceData = $service->toArray();
        if ($service->image) {
            $serviceData['image_url'] = Storage::url($service->image);
        }

        return response()->json([
            'success' => true,
            'message' => 'Service added successfully',
            'data' => $serviceData
        ], 201);
    }

    /**
     * Update Service
     */
    public function update(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        // Debug: Log request data
        \Log::info('Update service request', [
            'id' => $id,
            'hasFile' => $request->hasFile('image'),
            'allFiles' => $request->allFiles(),
            'contentType' => $request->header('Content-Type'),
            'method' => $request->method(),
            'all' => $request->all()
        ]);

        // Handle PUT requests with file uploads - Laravel doesn't parse files in PUT requests
        // So we need to manually parse it first before validation
        $hasImageFile = $request->hasFile('image');
        $imageFile = null;
        
        // For PUT requests with multipart/form-data, manually parse the file BEFORE validation
        if (!$hasImageFile && $request->method() === 'PUT' && str_contains($request->header('Content-Type', ''), 'multipart/form-data')) {
            $imageFile = $this->parseFileFromPutRequest($request, 'image');
            if ($imageFile) {
                $hasImageFile = true;
                // Merge the file into the request so validation works
                $request->files->set('image', $imageFile);
                // Also add to request input for validation
                $request->merge(['image' => $imageFile]);
                \Log::info('File extracted from PUT request manually (before validation)', [
                    'filename' => $imageFile->getClientOriginalName(),
                    'size' => $imageFile->getSize(),
                    'mimeType' => $imageFile->getMimeType()
                ]);
            }
        }

        $validationRules = [
            'name' => 'sometimes|required|string|max:255',
            'path' => 'sometimes|required|string|unique:services,path,' . $id . '|regex:/^\/[a-zA-Z0-9\/-]+$/',
            'is_coming_soon' => 'nullable|boolean',
            'cost' => 'nullable|numeric|min:0',
        ];

        // For manually parsed files, validate manually instead of using Laravel's validator
        // Laravel's 'uploaded' rule fails for manually created UploadedFile instances
        if ($imageFile) {
            // Manual validation for PUT request files
            $errors = $this->validateManuallyParsedFile($imageFile);
            if (!empty($errors)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => ['image' => $errors]
                ], 422);
            }
        } elseif ($hasImageFile || $request->method() === 'POST' || $request->method() === 'PATCH' || $request->method() === 'PUT') {
            // Standard validation for POST/PATCH requests
            $validationRules['image'] = 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048';
        }

        $validator = Validator::make($request->all(), $validationRules);

        if ($validator->fails()) {
            \Log::warning('Service update validation failed', [
                'errors' => $validator->errors()->toArray(),
                'request_data' => $request->all()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $service = Service::find($id);

        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Service not found'
            ], 404);
        }

        // Use the already parsed file (if PUT) or check request file
        if ($hasImageFile) {
            try {
                // Use manually parsed file for PUT requests, otherwise use request file
                $image = $imageFile ?: $request->file('image');
                $isManuallyParsed = (bool)$imageFile;
                
                // Debug logging
                \Log::info('Image upload attempt (update)', [
                    'hasFile' => $request->hasFile('image'),
                    'isValid' => $image->isValid(),
                    'originalName' => $image->getClientOriginalName(),
                    'size' => $image->getSize(),
                    'mimeType' => $image->getMimeType(),
                    'source' => $isManuallyParsed ? 'PUT manual parse' : 'request file',
                    'error' => $image->getError()
                ]);
                
                // For manually parsed files, isValid() might return false even if file is valid
                // So we check file existence and size instead
                if (!$isManuallyParsed && !$image->isValid()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid image file'
                    ], 422);
                }
                
                // For manually parsed files, verify the file exists and has content
                if ($isManuallyParsed) {
                    if (!file_exists($image->getPathname()) || filesize($image->getPathname()) === 0) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Invalid image file: file is empty or does not exist'
                        ], 422);
                    }
                    
                    // Verify it's actually an image
                    $imageInfo = @getimagesize($image->getPathname());
                    if ($imageInfo === false) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Invalid image file: not a valid image format'
                        ], 422);
                    }
                }
                
                // Delete old image if exists
                if ($service->image && Storage::disk('public')->exists($service->image)) {
                    Storage::disk('public')->delete($service->image);
                }

                // Ensure the directory exists
                $directory = storage_path('app/public/services');
                if (!file_exists($directory)) {
                    mkdir($directory, 0755, true);
                }

                $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                
                // For manually parsed files (Symfony UploadedFile), use Storage facade directly
                // For regular files (Laravel UploadedFile), use storeAs method
                if ($isManuallyParsed) {
                    // Read file content and store using Storage facade
                    $fileContent = file_get_contents($image->getPathname());
                    $imagePath = 'services/' . $imageName;
                    Storage::disk('public')->put($imagePath, $fileContent);
                    
                    // Clean up temporary file
                    @unlink($image->getPathname());
                } else {
                    // Use Laravel's storeAs method for regular uploads
                    $imagePath = $image->storeAs('services', $imageName, 'public');
                }
                
                \Log::info('Image stored (update)', [
                    'imagePath' => $imagePath,
                    'fullPath' => storage_path('app/public/' . $imagePath),
                    'exists' => Storage::disk('public')->exists($imagePath),
                    'isManuallyParsed' => $isManuallyParsed
                ]);
                
                // Verify file was saved
                if (!$imagePath || !Storage::disk('public')->exists($imagePath)) {
                    \Log::error('Image upload failed during update', [
                        'imagePath' => $imagePath,
                        'directory' => $directory,
                        'exists' => $imagePath ? Storage::disk('public')->exists($imagePath) : false
                    ]);
                    
                    return response()->json([
                        'success' => false,
                        'message' => 'Image file was not saved successfully'
                    ], 422);
                }
                
                $updateData = array_merge($request->only(['name', 'path', 'is_coming_soon', 'cost']), ['image' => $imagePath]);
            } catch (\Exception $e) {
                \Log::error('Image upload exception during update', [
                    'message' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Image upload failed: ' . $e->getMessage()
                ], 422);
            }
        } else {
            $updateData = $request->only(['name', 'path', 'is_coming_soon', 'cost']);
        }

        $service->update($updateData);

        // Format response with image URL
        $serviceData = $service->fresh()->toArray();
        if ($service->fresh()->image) {
            $serviceData['image_url'] = Storage::url($service->fresh()->image);
        }

        return response()->json([
            'success' => true,
            'message' => 'Service updated successfully',
            'data' => $serviceData
        ], 200);
    }

    /**
     * Delete Service
     */
    public function destroy(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $service = Service::find($id);

        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Service not found'
            ], 404);
        }

        // Delete image file if exists
        if ($service->image && Storage::disk('public')->exists($service->image)) {
            Storage::disk('public')->delete($service->image);
        }

        $service->delete();

        return response()->json([
            'success' => true,
            'message' => 'Service deleted successfully'
        ], 200);
    }
}
