<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Admin;
use App\Models\TopUpRequest;
use App\Models\ServiceTransaction;
use App\Models\Service;
use App\Models\ReferralRelationship;
use App\Models\ReferralEarning;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\PersonalAccessToken;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Get User Recharge Report
     * Shows detailed recharge history for a specific user or all users
     */
    public function userRechargeReport(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $query = TopUpRequest::with(['user:id,email,phone_number,name,user_type', 'upi:id,upi_id,name']);

        // Filter by user_id if provided
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        // Filter by approved date range
        if ($request->has('approved_start_date')) {
            $query->whereDate('approved_at', '>=', $request->approved_start_date);
        }

        if ($request->has('approved_end_date')) {
            $query->whereDate('approved_at', '<=', $request->approved_end_date);
        }

        // Sort options
        $sortBy = $request->input('sort_by', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->input('per_page', 15);
        $perPage = min(max($perPage, 1), 100);

        $recharges = $query->paginate($perPage);

        // Calculate summary statistics
        $summaryQuery = TopUpRequest::query();
        
        if ($request->has('user_id')) {
            $summaryQuery->where('user_id', $request->user_id);
        }
        
        if ($request->has('start_date')) {
            $summaryQuery->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->has('end_date')) {
            $summaryQuery->whereDate('created_at', '<=', $request->end_date);
        }

        $summary = [
            'total_requests' => $summaryQuery->count(),
            'total_amount' => (float) $summaryQuery->sum('amount'),
            'total_credited' => (float) $summaryQuery->where('status', 'approved')->sum('final_credited_amount'),
            'pending_requests' => (clone $summaryQuery)->where('status', 'pending')->count(),
            'approved_requests' => (clone $summaryQuery)->where('status', 'approved')->count(),
            'rejected_requests' => (clone $summaryQuery)->where('status', 'rejected')->count(),
            'pending_amount' => (float) (clone $summaryQuery)->where('status', 'pending')->sum('amount'),
            'approved_amount' => (float) (clone $summaryQuery)->where('status', 'approved')->sum('amount'),
        ];

        return response()->json([
            'success' => true,
            'summary' => $summary,
            'data' => $recharges->items(),
            'pagination' => [
                'current_page' => $recharges->currentPage(),
                'per_page' => $recharges->perPage(),
                'total' => $recharges->total(),
                'last_page' => $recharges->lastPage(),
            ]
        ], 200);
    }

    /**
     * Get User Service Report
     * Shows detailed service usage history for a specific user or all users
     */
    public function userServiceReport(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $query = ServiceTransaction::with([
            'user:id,email,phone_number,name,user_type',
            'service:id,name,path'
        ]);

        // Filter by user_id if provided
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Filter by service_id
        if ($request->has('service_id')) {
            $query->where('service_id', $request->service_id);
        }

        // Filter by service_type
        if ($request->has('service_type')) {
            $query->where('service_type', $request->service_type);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        // Sort options
        $sortBy = $request->input('sort_by', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->input('per_page', 15);
        $perPage = min(max($perPage, 1), 100);

        $services = $query->paginate($perPage);

        // Calculate summary statistics
        $summaryQuery = ServiceTransaction::query();
        
        if ($request->has('user_id')) {
            $summaryQuery->where('user_id', $request->user_id);
        }
        
        if ($request->has('service_id')) {
            $summaryQuery->where('service_id', $request->service_id);
        }
        
        if ($request->has('service_type')) {
            $summaryQuery->where('service_type', $request->service_type);
        }
        
        if ($request->has('start_date')) {
            $summaryQuery->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->has('end_date')) {
            $summaryQuery->whereDate('created_at', '<=', $request->end_date);
        }

        $summary = [
            'total_transactions' => $summaryQuery->count(),
            'total_amount' => (float) $summaryQuery->sum('amount'),
            'successful_transactions' => (clone $summaryQuery)->where('status', 'success')->count(),
            'failed_transactions' => (clone $summaryQuery)->where('status', 'failed')->count(),
            'pending_transactions' => (clone $summaryQuery)->where('status', 'pending')->count(),
            'successful_amount' => (float) (clone $summaryQuery)->where('status', 'success')->sum('amount'),
            'failed_amount' => (float) (clone $summaryQuery)->where('status', 'failed')->sum('amount'),
        ];

        // Service type breakdown
        $serviceTypeBreakdown = (clone $summaryQuery)
            ->select('service_type', DB::raw('COUNT(*) as count'), DB::raw('SUM(amount) as total_amount'))
            ->groupBy('service_type')
            ->get()
            ->map(function ($item) {
                return [
                    'service_type' => $item->service_type,
                    'count' => $item->count,
                    'total_amount' => (float) $item->total_amount,
                ];
            });

        return response()->json([
            'success' => true,
            'summary' => $summary,
            'service_type_breakdown' => $serviceTypeBreakdown,
            'data' => $services->items(),
            'pagination' => [
                'current_page' => $services->currentPage(),
                'per_page' => $services->perPage(),
                'total' => $services->total(),
                'last_page' => $services->lastPage(),
            ]
        ], 200);
    }

    /**
     * Get User Complete Report
     * Shows both recharge and service reports for a specific user
     */
    public function userCompleteReport(Request $request, $userId)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $user = User::find($userId);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        // Date range filters
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        // Recharge Statistics
        $rechargeQuery = TopUpRequest::where('user_id', $userId);
        if ($startDate) {
            $rechargeQuery->whereDate('created_at', '>=', $startDate);
        }
        if ($endDate) {
            $rechargeQuery->whereDate('created_at', '<=', $endDate);
        }

        $rechargeStats = [
            'total_requests' => $rechargeQuery->count(),
            'total_amount' => (float) $rechargeQuery->sum('amount'),
            'total_credited' => (float) $rechargeQuery->where('status', 'approved')->sum('final_credited_amount'),
            'pending' => $rechargeQuery->where('status', 'pending')->count(),
            'approved' => $rechargeQuery->where('status', 'approved')->count(),
            'rejected' => $rechargeQuery->where('status', 'rejected')->count(),
        ];

        // Service Statistics
        $serviceQuery = ServiceTransaction::where('user_id', $userId);
        if ($startDate) {
            $serviceQuery->whereDate('created_at', '>=', $startDate);
        }
        if ($endDate) {
            $serviceQuery->whereDate('created_at', '<=', $endDate);
        }

        $serviceStats = [
            'total_transactions' => $serviceQuery->count(),
            'total_amount' => (float) $serviceQuery->sum('amount'),
            'successful' => $serviceQuery->where('status', 'success')->count(),
            'failed' => $serviceQuery->where('status', 'failed')->count(),
            'pending' => $serviceQuery->where('status', 'pending')->count(),
        ];

        // Service type breakdown
        $serviceTypeBreakdown = (clone $serviceQuery)
            ->select('service_type', DB::raw('COUNT(*) as count'), DB::raw('SUM(amount) as total_amount'))
            ->groupBy('service_type')
            ->get()
            ->map(function ($item) {
                return [
                    'service_type' => $item->service_type,
                    'count' => $item->count,
                    'total_amount' => (float) $item->total_amount,
                ];
            });

        // Recent recharges (last 10)
        $recentRecharges = TopUpRequest::where('user_id', $userId)
            ->with(['upi:id,upi_id,name'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Recent services (last 10)
        $recentServices = ServiceTransaction::where('user_id', $userId)
            ->with(['service:id,name,path'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'user' => [
                'id' => $user->id,
                'email' => $user->email,
                'phone_number' => $user->phone_number,
                'name' => $user->name,
                'user_type' => $user->user_type,
                'wallet_balance' => (float) $user->wallet_balance,
                'is_verified' => $user->is_verified,
            ],
            'recharge_statistics' => $rechargeStats,
            'service_statistics' => $serviceStats,
            'service_type_breakdown' => $serviceTypeBreakdown,
            'recent_recharges' => $recentRecharges,
            'recent_services' => $recentServices,
        ], 200);
    }

    /**
     * Get Referral Report
     * Shows referral statistics including level 1 referrals, downline, and their recharge activity
     */
    public function referralReport(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $query = User::query();

        // Filter by user_id if provided
        if ($request->has('user_id')) {
            $query->where('id', $request->user_id);
        }

        // Filter by user_type
        if ($request->has('user_type')) {
            $query->where('user_type', $request->user_type);
        }

        // Filter by users who have referrals
        if ($request->has('has_referrals') && $request->has_referrals == 'true') {
            $query->whereHas('referralRelationships');
        }

        // Sort options
        $sortBy = $request->input('sort_by', 'total_referrals'); // total_referrals, total_downline_recharge, total_earnings
        $sortOrder = $request->input('sort_order', 'desc');

        // Pagination
        $perPage = $request->input('per_page', 15);
        $perPage = min(max($perPage, 1), 100);

        $users = $query->paginate($perPage);

        // Build referral statistics for each user
        $referralData = $users->map(function ($user) use ($request) {
            // Level 1 referrals count
            $level1Count = ReferralRelationship::where('referrer_id', $user->id)
                ->where('level', 1)
                ->count();

            // Total downline count (all levels)
            $totalDownlineCount = ReferralRelationship::where('referrer_id', $user->id)
                ->count();

            // Get all referred user IDs (all levels)
            $referredUserIds = ReferralRelationship::where('referrer_id', $user->id)
                ->pluck('referred_id')
                ->toArray();

            // Calculate downline recharge statistics
            $downlineRechargeQuery = TopUpRequest::whereIn('user_id', $referredUserIds)
                ->where('status', 'approved');

            // Filter by date range if provided
            if ($request->has('start_date')) {
                $downlineRechargeQuery->whereDate('approved_at', '>=', $request->start_date);
            }
            if ($request->has('end_date')) {
                $downlineRechargeQuery->whereDate('approved_at', '<=', $request->end_date);
            }

            $downlineRechargeStats = [
                'total_recharge_amount' => (float) $downlineRechargeQuery->sum('amount'),
                'total_recharge_count' => $downlineRechargeQuery->count(),
                'total_credited_amount' => (float) $downlineRechargeQuery->sum('final_credited_amount'),
            ];

            // Get level-wise breakdown
            $levelBreakdown = [];
            for ($level = 1; $level <= 5; $level++) {
                $levelReferredIds = ReferralRelationship::where('referrer_id', $user->id)
                    ->where('level', $level)
                    ->pluck('referred_id')
                    ->toArray();

                if (count($levelReferredIds) > 0) {
                    $levelRechargeQuery = TopUpRequest::whereIn('user_id', $levelReferredIds)
                        ->where('status', 'approved');

                    if ($request->has('start_date')) {
                        $levelRechargeQuery->whereDate('approved_at', '>=', $request->start_date);
                    }
                    if ($request->has('end_date')) {
                        $levelRechargeQuery->whereDate('approved_at', '<=', $request->end_date);
                    }

                    $levelBreakdown[] = [
                        'level' => $level,
                        'referral_count' => count($levelReferredIds),
                        'recharge_amount' => (float) $levelRechargeQuery->sum('amount'),
                        'recharge_count' => $levelRechargeQuery->count(),
                    ];
                }
            }

            // Referral earnings
            $earningsQuery = ReferralEarning::where('referrer_id', $user->id);
            if ($request->has('start_date')) {
                $earningsQuery->whereDate('created_at', '>=', $request->start_date);
            }
            if ($request->has('end_date')) {
                $earningsQuery->whereDate('created_at', '<=', $request->end_date);
            }

            $earningsStats = [
                'total_earnings' => (float) (clone $earningsQuery)->where('status', 'credited')->sum('commission_amount'),
                'pending_earnings' => (float) (clone $earningsQuery)->where('status', 'pending')->sum('commission_amount'),
                'total_earning_count' => $earningsQuery->count(),
            ];

            return [
                'user_id' => $user->id,
                'user' => [
                    'id' => $user->id,
                    'email' => $user->email,
                    'phone_number' => $user->phone_number,
                    'name' => $user->name,
                    'user_type' => $user->user_type,
                    'referral_code' => $user->referral_code,
                ],
                'referral_statistics' => [
                    'level_1_referrals' => $level1Count,
                    'total_downline_count' => $totalDownlineCount,
                ],
                'downline_recharge' => $downlineRechargeStats,
                'level_breakdown' => $levelBreakdown,
                'referral_earnings' => $earningsStats,
            ];
        });

        // Sort the data
        if ($sortBy === 'total_downline_recharge') {
            $referralData = $referralData->sortByDesc(function ($item) {
                return $item['downline_recharge']['total_recharge_amount'];
            })->values();
        } elseif ($sortBy === 'total_earnings') {
            $referralData = $referralData->sortByDesc(function ($item) {
                return $item['referral_earnings']['total_earnings'];
            })->values();
        } else {
            $referralData = $referralData->sortByDesc(function ($item) {
                return $item['referral_statistics']['total_downline_count'];
            })->values();
        }

        // Calculate overall summary
        $summary = [
            'total_users_with_referrals' => User::whereHas('referralRelationships')->count(),
            'total_referrals' => ReferralRelationship::count(),
            'total_level_1_referrals' => ReferralRelationship::where('level', 1)->count(),
            'total_downline_recharge' => (float) TopUpRequest::whereIn('user_id', 
                ReferralRelationship::pluck('referred_id')->toArray()
            )->where('status', 'approved')->sum('amount'),
            'total_referral_earnings' => (float) ReferralEarning::where('status', 'credited')->sum('commission_amount'),
        ];

        return response()->json([
            'success' => true,
            'summary' => $summary,
            'data' => $referralData->all(),
            'pagination' => [
                'current_page' => $users->currentPage(),
                'per_page' => $users->perPage(),
                'total' => $users->total(),
                'last_page' => $users->lastPage(),
            ]
        ], 200);
    }

    /**
     * Get User-Specific Referral Report
     * Detailed referral report for a specific user
     */
    public function userReferralReport(Request $request, $userId)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $user = User::find($userId);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        // Date range filters
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        // Level 1 referrals
        $level1Query = ReferralRelationship::where('referrer_id', $user->id)
            ->where('level', 1)
            ->with(['referred:id,email,phone_number,name,user_type,created_at']);

        $level1Referrals = $level1Query->get();

        // Total downline (all levels)
        $totalDownlineQuery = ReferralRelationship::where('referrer_id', $user->id);
        $totalDownline = $totalDownlineQuery->count();

        // Get all referred user IDs (all levels)
        $referredUserIds = ReferralRelationship::where('referrer_id', $user->id)
            ->pluck('referred_id')
            ->toArray();

        // Downline recharge statistics
        $downlineRechargeQuery = TopUpRequest::whereIn('user_id', $referredUserIds)
            ->where('status', 'approved')
            ->with(['user:id,email,phone_number,name']);

        if ($startDate) {
            $downlineRechargeQuery->whereDate('approved_at', '>=', $startDate);
        }
        if ($endDate) {
            $downlineRechargeQuery->whereDate('approved_at', '<=', $endDate);
        }

        $downlineRecharges = $downlineRechargeQuery->orderBy('approved_at', 'desc')->get();

        $downlineRechargeStats = [
            'total_recharge_amount' => (float) $downlineRecharges->sum('amount'),
            'total_recharge_count' => $downlineRecharges->count(),
            'total_credited_amount' => (float) $downlineRecharges->sum('final_credited_amount'),
        ];

        // Level-wise breakdown
        $levelBreakdown = [];
        for ($level = 1; $level <= 5; $level++) {
            $levelRelationships = ReferralRelationship::where('referrer_id', $user->id)
                ->where('level', $level)
                ->with(['referred:id,email,phone_number,name,user_type'])
                ->get();

            if ($levelRelationships->count() > 0) {
                $levelReferredIds = $levelRelationships->pluck('referred_id')->toArray();
                
                $levelRechargeQuery = TopUpRequest::whereIn('user_id', $levelReferredIds)
                    ->where('status', 'approved');

                if ($startDate) {
                    $levelRechargeQuery->whereDate('approved_at', '>=', $startDate);
                }
                if ($endDate) {
                    $levelRechargeQuery->whereDate('approved_at', '<=', $endDate);
                }

                $levelBreakdown[] = [
                    'level' => $level,
                    'referral_count' => $levelRelationships->count(),
                    'referrals' => $levelRelationships->map(function ($rel) {
                        return [
                            'user_id' => $rel->referred->id,
                            'email' => $rel->referred->email,
                            'phone_number' => $rel->referred->phone_number,
                            'name' => $rel->referred->name,
                            'user_type' => $rel->referred->user_type,
                            'joined_at' => $rel->referred->created_at,
                        ];
                    }),
                    'recharge_amount' => (float) $levelRechargeQuery->sum('amount'),
                    'recharge_count' => $levelRechargeQuery->count(),
                ];
            }
        }

        // Referral earnings
        $earningsQuery = ReferralEarning::where('referrer_id', $user->id)
            ->with(['referred:id,email,phone_number,name']);

        if ($startDate) {
            $earningsQuery->whereDate('created_at', '>=', $startDate);
        }
        if ($endDate) {
            $earningsQuery->whereDate('created_at', '<=', $endDate);
        }

        $earnings = $earningsQuery->orderBy('created_at', 'desc')->get();

        $earningsStats = [
            'total_earnings' => (float) (clone $earningsQuery)->where('status', 'credited')->sum('commission_amount'),
            'pending_earnings' => (float) (clone $earningsQuery)->where('status', 'pending')->sum('commission_amount'),
            'total_earning_count' => $earnings->count(),
        ];

        return response()->json([
            'success' => true,
            'user' => [
                'id' => $user->id,
                'email' => $user->email,
                'phone_number' => $user->phone_number,
                'name' => $user->name,
                'user_type' => $user->user_type,
                'referral_code' => $user->referral_code,
            ],
            'referral_statistics' => [
                'level_1_referrals' => $level1Referrals->count(),
                'total_downline_count' => $totalDownline,
            ],
            'level_1_referrals' => $level1Referrals->map(function ($rel) {
                return [
                    'user_id' => $rel->referred->id,
                    'email' => $rel->referred->email,
                    'phone_number' => $rel->referred->phone_number,
                    'name' => $rel->referred->name,
                    'user_type' => $rel->referred->user_type,
                    'joined_at' => $rel->referred->created_at,
                ];
            }),
            'level_breakdown' => $levelBreakdown,
            'downline_recharge' => $downlineRechargeStats,
            'downline_recharges' => $downlineRecharges->map(function ($recharge) {
                return [
                    'id' => $recharge->id,
                    'user' => $recharge->user ? [
                        'id' => $recharge->user->id,
                        'email' => $recharge->user->email,
                        'phone_number' => $recharge->user->phone_number,
                        'name' => $recharge->user->name,
                    ] : null,
                    'amount' => (float) $recharge->amount,
                    'final_credited_amount' => (float) $recharge->final_credited_amount,
                    'approved_at' => $recharge->approved_at,
                ];
            }),
            'referral_earnings' => $earningsStats,
            'earnings' => $earnings->map(function ($earning) {
                return [
                    'id' => $earning->id,
                    'referred_user' => $earning->referred ? [
                        'id' => $earning->referred->id,
                        'email' => $earning->referred->email,
                        'name' => $earning->referred->name,
                    ] : null,
                    'level' => $earning->level,
                    'earning_type' => $earning->earning_type,
                    'amount' => (float) $earning->amount,
                    'commission_amount' => (float) $earning->commission_amount,
                    'status' => $earning->status,
                    'created_at' => $earning->created_at,
                    'credited_at' => $earning->credited_at,
                ];
            }),
        ], 200);
    }
}

