<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\ReferralSetting;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;

class ReferralController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Get Referral Settings
     */
    public function getSettings(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $settings = ReferralSetting::getActive();

        if (!$settings) {
            return response()->json([
                'success' => true,
                'data' => null,
                'message' => 'No referral settings configured'
            ], 200);
        }

        return response()->json([
            'success' => true,
            'data' => $settings
        ], 200);
    }

    /**
     * Create or Update Referral Settings
     */
    public function updateSettings(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'referral_levels' => 'required|integer|min:1|max:10',
            'level_commissions' => 'required|array',
            'level_commissions.*' => 'required|numeric|min:0',
            'commission_type' => 'required|in:percentage,fixed',
            'signup_bonus' => 'nullable|numeric|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Validate level_commissions has entries for all levels
        $levels = $request->referral_levels;
        $commissions = $request->level_commissions;
        
        for ($i = 1; $i <= $levels; $i++) {
            if (!isset($commissions[$i])) {
                return response()->json([
                    'success' => false,
                    'message' => "Commission rate for level {$i} is required"
                ], 422);
            }
        }

        // Deactivate existing settings
        ReferralSetting::where('is_active', true)->update(['is_active' => false]);

        // Create new settings
        $settings = ReferralSetting::create([
            'referral_levels' => $levels,
            'level_commissions' => $commissions,
            'commission_type' => $request->commission_type,
            'signup_bonus' => $request->signup_bonus ?? 0,
            'is_active' => $request->is_active ?? true,
            'created_by' => $admin->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Referral settings updated successfully',
            'data' => $settings
        ], 200);
    }

    /**
     * Get All Referral Statistics
     */
    public function getStatistics(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $stats = [
            'total_referrals' => \App\Models\User::whereNotNull('referred_by')->count(),
            'total_earnings' => \App\Models\ReferralEarning::where('status', 'credited')->sum('commission_amount'),
            'pending_earnings' => \App\Models\ReferralEarning::where('status', 'pending')->sum('commission_amount'),
            'total_signup_bonuses' => \App\Models\ReferralEarning::where('earning_type', 'signup')
                ->where('status', 'credited')
                ->sum('commission_amount'),
            'total_topup_commissions' => \App\Models\ReferralEarning::where('earning_type', 'topup')
                ->where('status', 'credited')
                ->sum('commission_amount'),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ], 200);
    }
}
