<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Offer;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;

class OfferController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Get All Offers
     */
    public function index(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $offers = Offer::with('admin')->orderBy('order', 'asc')->orderBy('created_at', 'desc')->get();

        return response()->json([
            'success' => true,
            'data' => $offers
        ], 200);
    }

    /**
     * Get Single Offer
     */
    public function show(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $offer = Offer::with('admin')->find($id);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Offer not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $offer
        ], 200);
    }

    /**
     * Add New Offer
     */
    public function store(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image_url' => 'nullable|string|url',
            'offer_type' => 'nullable|in:service_discount,signup_bonus,referral_bonus,cashback,general',
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
            'discount_amount' => 'nullable|numeric|min:0',
            'cashback_amount' => 'nullable|numeric|min:0',
            'promo_code' => 'nullable|string|max:50|unique:offers,promo_code',
            'valid_from' => 'required|date',
            'valid_to' => 'required|date|after_or_equal:valid_from',
            'link' => 'nullable|string|url',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $offer = Offer::create([
            'title' => $request->title,
            'description' => $request->description,
            'image_url' => $request->image_url,
            'offer_type' => $request->offer_type ?? 'general',
            'discount_percentage' => $request->discount_percentage,
            'discount_amount' => $request->discount_amount,
            'cashback_amount' => $request->cashback_amount,
            'promo_code' => $request->promo_code,
            'valid_from' => $request->valid_from,
            'valid_to' => $request->valid_to,
            'link' => $request->link,
            'order' => $request->order ?? 0,
            'is_active' => $request->is_active ?? true,
            'created_by' => $admin->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Offer added successfully',
            'data' => $offer->load('admin')
        ], 201);
    }

    /**
     * Update Offer
     */
    public function update(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'image_url' => 'nullable|string|url',
            'offer_type' => 'nullable|in:service_discount,signup_bonus,referral_bonus,cashback,general',
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
            'discount_amount' => 'nullable|numeric|min:0',
            'cashback_amount' => 'nullable|numeric|min:0',
            'promo_code' => 'nullable|string|max:50|unique:offers,promo_code,' . $id,
            'valid_from' => 'sometimes|required|date',
            'valid_to' => 'sometimes|required|date|after_or_equal:valid_from',
            'link' => 'nullable|string|url',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $offer = Offer::find($id);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Offer not found'
            ], 404);
        }

        $offer->update($request->only([
            'title', 'description', 'image_url', 'offer_type',
            'discount_percentage', 'discount_amount', 'cashback_amount',
            'promo_code', 'valid_from', 'valid_to', 'link', 'order', 'is_active'
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Offer updated successfully',
            'data' => $offer->fresh()->load('admin')
        ], 200);
    }

    /**
     * Delete Offer
     */
    public function destroy(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $offer = Offer::find($id);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Offer not found'
            ], 404);
        }

        $offer->delete();

        return response()->json([
            'success' => true,
            'message' => 'Offer deleted successfully'
        ], 200);
    }
}
